/*
    This script class allows to read data from wave files.
    
    In order to use the class create an instance and call method 'Read(...)' in order to 
    open an read in the wave file.
    After that use the interface methods in order to access the actual wave data in the file.
*/

#include "GlobalDefinitions.as"

// Constants needed to identify certain sections in a wave file
const string CHUNK_ID_RIFF = "RIFF";
const string RIFF_TYPE = "WAVE";
const string FORMAT_SECTION = "fmt ";
const string DATA_SECTION = "data";

// The PCM format is the only one supprted by this script class.
const uint16 FORMAT_TAG_PCM = 0x0001;

class WaveFile
{
    // Read in the whole wave file. Return true on success.
    bool Read(string fileName)
    {
        bool success = mFile.Open(fileName, FILE_MODE_READ | FILE_SHARE_DENY_NONE | FILE_TYPE_BINARY);
                                    
        success = success && mFile.GetLength(mFileLen);
        
        if (success && (mFileLen != 0))
        {
            if (DEBUG) Trace("File Length =  " + mFileLen + "\r\n");
            
            uint read = mFile.Read(mFileContent, mFileLen);
            
            success = (read == mFileLen);
            if (DEBUG) Trace("Read " + read + " Bytes from file\r\n");
        }
        
        success = success && Analyse();
                                    
        return success;
    }
    
    // Get number of channels available in the wave file (1 for MONO, 2 for STEREO)
    uint16 GetChannelCount()
    {
        return mChannels;
    }
    
    // Get number of data points per channel
    uint GetSampleCountPerChannel()
    {
        uint sampleCount = 0;
        
        if (mBlockAlign != 0)
        {
            sampleCount = uint(mChunkSizeData / mBlockAlign);
        }
        
        return sampleCount;
    }
    
    // Get the next data point for both left and right channel. If the file has one channel only 
    // (i.e. MONO) then the left channel will contain the data point and the right channel will be 
    // returned as 0.
    bool GetNextSample(int &out left, int &out right)
    {
        bool success = mChannels != 0;
        uint16 sampleWidth = mBlockAlign / mChannels;
        
        if (success)
        {
            switch (sampleWidth)
            {
                case 1:
                    success = GetNextUINT8Sample(left, right);
                    break;
                    
                case 2:
                    success = GetNextINT16Sample(left, right);
                    break;    
                    
                case 4:
                    success = GetNextINT32Sample(left, right);
                    break;  
                    
                 default:
                    success = false;  
            }
        }
        
        return success;
    }
    
    // Restart retrieving samples from the beginning of the wave file.
    void Rewind()
    {
        mNextSampleIdx = mSampleStartIdx;
    }
    
        
    
    
    
// ----------- Private methods below this line. Don't call them from outside the class--------------------------------    
    bool GetNextUINT8Sample(int &out left, int &out right)
    {
        uint8 sample;
        uint idx = UINT8FromBuffer(mFileContent, mNextSampleIdx, sample);
        bool success = (idx > 0);
        
        if (success)
        {
            left = int(uint(sample));
            
            if (mChannels > 1)
            {
                idx = UINT8FromBuffer(mFileContent, idx, sample);
                success = (idx > 0);
                if (success)
                {
                    right = int(uint(sample));
                }
            }
            else
            {
                right = 0;
            }
            mNextSampleIdx += mBlockAlign;
        }
        
        return success;
    }
    
    bool GetNextINT16Sample(int &out left, int &out right)
    {
        uint16 sample;
        uint idx = UINT16FromBuffer(mFileContent, mNextSampleIdx, sample);
        bool success = (idx > 0);
        
        if (success)
        {
            left = int(int16(sample));
            
            if (mChannels > 1)
            {
                idx = UINT16FromBuffer(mFileContent, idx, sample);
                success = (idx > 0);
                if (success)
                {
                    right = int(int16(sample));
                }
            }
            else
            {
                right = 0;
            }
            mNextSampleIdx += mBlockAlign;
        }
        
        return success;
    }
    
    bool GetNextINT32Sample(int &out left, int &out right)
    {
        uint32 sample;
        uint idx = UINT32FromBuffer(mFileContent, mNextSampleIdx, sample);
        bool success = (idx > 0);
        
        if (success)
        {
            left = int(sample);
            
            if (mChannels > 1)
            {
                idx = UINT32FromBuffer(mFileContent, idx, sample);
                success = (idx > 0);
                if (success)
                {
                    right = int(sample);
                }
            }
            else
            {
                right = 0;
            }
            mNextSampleIdx += mBlockAlign;
        }
        
        return success;
    }
    
    
    

    bool Analyse()
    {
        int idx = mFileContent.findFirst(CHUNK_ID_RIFF, 0);
        bool success = idx >= 0;
        
        if (success)
        {
            if (DEBUG) Trace("RIFF idx = " + idx + "\r\n");
            idx += CHUNK_ID_RIFF.length();
            idx = UINT32FromBuffer(mFileContent, idx, mChunkSizeRiff);
            success = (idx > 0);
            if (DEBUG) Trace("Chunk size RIFF: " + mChunkSizeRiff + "\r\n");
        }
        if (success)
        {
            idx = mFileContent.findFirst(RIFF_TYPE, 0);
            if (DEBUG) Trace("RIFF Type idx = " + idx + "\r\n");
            success = (idx >= 0);
        }
        if (success)
        {
            idx = mFileContent.findFirst(FORMAT_SECTION, idx);
            success = (idx >= 0);
            if (DEBUG) Trace("FMT section idx = " + idx + "\r\n");
            idx += FORMAT_SECTION.length();
        }
        if (success)
        {
            success = false;
            idx = UINT32FromBuffer(mFileContent, idx, mChunkSizeFmt);
            if (DEBUG) Trace("Chunk size fmt: " + mChunkSizeFmt + "\r\n");
            if (idx > 0)
            {
                idx = UINT16FromBuffer(mFileContent, idx, mFormatTag);
                success = (idx > 0) && (mFormatTag == FORMAT_TAG_PCM);
            }
            if (success)
            {
                idx = UINT16FromBuffer(mFileContent, idx, mChannels);
                success = (idx > 0);
                if (DEBUG) Trace("Number of channels: " + mChannels + "\r\n");
            }
            if (success)
            {
                idx = UINT32FromBuffer(mFileContent, idx, mSamplesPerSec);
                success = (idx > 0);
                if (DEBUG) Trace("Samples Per Second: " + mSamplesPerSec + "\r\n");
            }
            if (success)
            {
                idx = UINT32FromBuffer(mFileContent, idx, mAvgBytesPerSec);
                success = (idx > 0);
                if (DEBUG) Trace("Avg Bytes Per Second: " + mAvgBytesPerSec + "\r\n");
            }
            if (success)
            {
                idx = UINT16FromBuffer(mFileContent, idx, mBlockAlign);
                success = (idx > 0);
                if (DEBUG) Trace("Block Align: " + mBlockAlign + "\r\n");
            }
            if (success)
            {
                idx = UINT16FromBuffer(mFileContent, idx, mBitsPerSample);
                success = (idx > 0);
                if (DEBUG) Trace("Bits Per Sample: " + mBitsPerSample + "\r\n");
            }
            if (success)
            {
                idx = mFileContent.findFirst(DATA_SECTION, idx);
                success = (idx >= 0);
                if (DEBUG) Trace("DATA section idx = " + idx + "\r\n");
                idx += DATA_SECTION.length();
            }
            if (success)
            {
                idx = UINT32FromBuffer(mFileContent, idx, mChunkSizeData);
                if (DEBUG) Trace("Chunk size data: " + mChunkSizeData + "\r\n");
                success = (idx > 0) && (mChunkSizeData > 0);
            }
            if (success)
            {
                mSampleStartIdx = idx;
                mNextSampleIdx = mSampleStartIdx;
                if (DEBUG) Trace("Samples start at index: " + mSampleStartIdx + "\r\n");
            }
        }
        
        return success;
    }
    
    
    uint UINT8FromBuffer(const string buf, uint idx, uint8& out value)
    {
        bool success = buf.length() >= idx + 1;
        uint nextIdx = 0;
        
        if (success)
        {
            value = buf[idx];
            nextIdx = idx + 1;
        }
        
        return nextIdx;
    }
    
    
    uint UINT16FromBuffer(const string buf, uint idx, uint16& out value)
    {
        bool success = buf.length() >= idx + 2;
        uint nextIdx = 0;
        
        if (success)
        {
            value = uint16(buf[idx]);
            value |= uint16(buf[idx + 1]) << 8;
            nextIdx = idx + 2;
        }
        
        return nextIdx;
    }
    
    uint UINT32FromBuffer(const string buf, uint idx, uint& out value)
    {
        uint16 lsb = 0;
        uint16 msb = 0;
        
        uint nextIdx = UINT16FromBuffer(buf, idx, lsb);
        
        if (nextIdx > 0)
        {
            nextIdx = UINT16FromBuffer(buf, nextIdx, msb);
        }
        
        if (nextIdx > 0)
        {
            value = uint(lsb);
            value |= uint(msb) << 16;
        }
        
        return nextIdx;
    }

    file mFile;
    string mFileContent;
    uint mFileLen;
    uint mChunkSizeRiff;
    uint mChunkSizeFmt;
    uint mChunkSizeData;
    
    uint mSampleStartIdx;
    uint mNextSampleIdx;
    
    uint16 mFormatTag;
    uint16 mChannels;
    uint mSamplesPerSec;
    uint mAvgBytesPerSec;
    uint16 mBlockAlign;
    uint16 mBitsPerSample;

}