#include "Position.as"

// Percentage overhead used when normalizing output neurons
const uint8 OUTPUT_NORMALIZE_OVERHEAD_PERCENT = 5; 
// Percentage overhead used when normalizing input neurons
const uint8 INPUT_NORMALIZE_OVERHEAD_PERCENT = 5; 

class NetEditor
{
    // Constructor
    NetEditor()
    {
        // Place the default origin on a grid point close to the middle of the drawing area
        SetOrigin(uint16(MAX_COORDINATE / 2), uint16(MAX_COORDINATE / 2));
        mHorSpacing = 1;
        mVertSpacing = 2;
    }
    
    // Set the origin for the net editor. Always place the origin on a grid point
    void SetOrigin(uint16 x, uint16 y)
    {
        uint16 gridX;
        uint16 gridY;
        GetGridPoint(x, y, gridX, gridY);
        mPos.Set(gridX, gridY);
    }
    
    // Set horizontal spacing in units of the adjusted grid width
    void SetHorizontalSpacing(uint16 spacing)
    {
        mHorSpacing = spacing;
    }
    
    // Set horizontal spacing in units of the adjusted grid width
    void SetVerticalSpacing(uint16 spacing)
    {
        mVertSpacing = spacing;
    }
    
    // Create a net with the given number of inputs and outputs at the current origin.
    // Place the output layer centered below the input layer and leave enough space for 
    // the given number of hidden layers.
    // Note: This method does not add any hidden neurons. 
    void CreateNet(uint16 inputCount, uint16 outputCount, uint16 maxPlannedHidLayerCount)
    {
        NewFile();
        Position pos = mPos;
        uint16 gridWidth = GetGridWidth();
        
        // Add all input neurons
        for (uint i = 1; i <= inputCount; i++)
        {
            AddInput(pos.x, pos.y, "In " + i);
            pos.Move((mHorSpacing + 1) * gridWidth, 0);
        }
        
        // Get X position for the output layer and set Y position to leave enough 
        // space for the planned max number of hidden layers
        pos.x = GetNewLayerXPos(outputCount);
        pos.y = mPos.y + ((mVertSpacing + 1) * (maxPlannedHidLayerCount + 1)) * gridWidth;      
                        
        for (uint i = 1; i <= outputCount; i++)
        {
            AddOutput(pos.x, pos.y, "Out " + i);
            pos.Move((mHorSpacing + 1) * gridWidth, 0);
        }
        
        // No hidden layers planned? Then connect the inputs to the outputs
        if (maxPlannedHidLayerCount == 0)
        {
            ConnectInputToOutput();
        }
        
        ZoomFit();
    }
    
    // Create a number of interconnected normal hidden layers for the net.
    // The array parameter handed over describes both numbers of hidden layers 
    // and neuron count per layer.
    void CreateHiddenLayers(uint16[] neuronCount)
    {
        uint16 layerCount = neuronCount.length();
        
        if (layerCount == 0)
            return;
            
        // Delete all eventually existing hidden layers.
        SelectHidden(false);
        DeleteSelectedObjects();
            
        // Add the hidden neurons in rows from output towards input
        for (uint16 i = 1; i <= layerCount; i++)
        {
            // Neuron layers will be added starting from the bottom
            uint16 count = neuronCount[layerCount - i];
            
            // Ensure that there will be at least one neuron in every layer
            if (count == 0)
            {
                count = 1;
            }
            
            AddHiddenNeurons(count);
            if (i == 1)
            { // The first layer (from bottom) has to be connected to the outputs 
              // of the net. It will become the context layer since the inputs of the 
              // neurons are not yet connected.
                ConnectUnresolvedToOutput();
            }
            else
            { // All other layers will be connected to the context layer. So the 
              // context layer becomes a new hidden normal layer and the unresolved 
              // neurons become the context layer.
                ConnectUnresolvedToContext();
            }
        }
        
        // Now connect inputs to context layer (which is the top most new layer
        // since these neurons have their inputs still unconnected)
        ConnectInputToContext();
        
        ZoomFit();
    }
    
    // Normalize the net using the currently active lesson. Return true on success.
    bool NormalizeNetWithActiveLesson()
    {
        uint count = GetInputCount();
        double actMin;
        double actMax;
        bool success = true;
        uint columnNum;
        
        for (uint i = 1; i <= count; i++)
        {
            columnNum = GetLessonInputColumnNum(i);
            if (columnNum == 0)
            {
                success = false;
                break;
            }
            GetLessonInputMinMax(columnNum, actMin, actMax);
            if (actMax > actMin)
            {
                NormalizeInputNeuron(i, actMin, actMax);
            }
            else
            {
                success = false;
                break;
            }
        }
        
        if (success)
        {
            count = GetOutputCount();
            for (uint i = 1; i <= count; i++)
            {
                columnNum = GetLessonOutputColumnNum(i);
                if (columnNum == 0)
                {
                    success = false;
                    break;
                }
                GetLessonOutputMinMax(columnNum, actMin, actMax);
                if (actMax > actMin)
                {
                    NormalizeOutputNeuron(i, actMin, actMax);
                }
                else
                {
                    success = false;
                    break;
                }
            }
        }
        
        return success;
    }
    
    // Connect all inputs to all outputs
    void ConnectInputToOutput()
    {
        SelectInput(false);
        ExtraSelection();
        SelectOutput(false);
        ConnectFromExtra();
        ClearExtraSelection();
        ClearSelection();
    }
    
    
    
//--------------------- private functions below this line --------------

    // Add a number of hidden neurons to the net.
    // The neurons will be placed so that they can build a new hidden layer, 
    // i.e. in a row above the highest existing normal hidden layer.
    // The neurons will be in the 'unresolved' layer afterwards since they 
    // are not yet connected. If there are already unresolved neurons present then 
    // the new ones will be added to the right and all unresolved neurons will be 
    // moved to the left.
    void AddHiddenNeurons(uint16 neuronCount)
    {
        uint16 unresolvedCount = GetUnresolvedCount();
        Position pos;
        uint16 gridWidth = GetGridWidth();
        
        pos.x = GetNewLayerXPos(neuronCount) + unresolvedCount * 2 * gridWidth;
        pos.y = GetNewHidLayerYPos();
        
        for (uint16 i = 0; i < neuronCount; i++)
        {
            AddHidden(pos.x, pos.y, "");
            pos.Move((mHorSpacing + 1) * gridWidth, 0);
        }
        
        // There were already more than one unresolved neurons. We have to move the whole new 
        // unresolved layer to the left.
        if (unresolvedCount > 1)
        {
            SelectUnresolved(false);
            MoveSelectedNeurons(-(neuronCount * mHorSpacing * gridWidth), 0);
            ClearSelection();
        }
    }
    
    // Connect all unresolved neurons to the output of the net
    void ConnectUnresolvedToOutput()
    {
        SelectUnresolved(false);
        ExtraSelection();
        SelectOutput(false);
        ConnectFromExtra();
        ClearExtraSelection();
        ClearSelection();
    }
    
    // Connect all unresolved neurons to the context layer of the net
    void ConnectUnresolvedToContext()
    {
        if (GetContextCount() > 0)
        {
            SelectUnresolved(false);
            ExtraSelection();
            SelectContext(false);
            ConnectFromExtra();
            ClearExtraSelection();
            ClearSelection();
        }
    }
    
    // Connect all input neurons to the context layer of the net
    void ConnectInputToContext()
    {
        if (GetInputCount() > 0)
        {
            SelectInput(false);
            ExtraSelection();
            SelectContext(false);
            ConnectFromExtra();
            ClearExtraSelection();
            ClearSelection();
        }
    }
    
    // Get the X-Position for a new layer with the planned given number of neurons 
    // so that it will be positioned horizontally centered with respect to the input 
    // neurons. 
    uint16 GetNewLayerXPos(uint16 neuronCount)
    {
        uint16 newX;
        uint16 inCount = GetInputCount();
                
        if (inCount > 0)
        {
            // Get position of left most input neuron.
            Position pos;
            uint16 gridWidth = GetGridWidth();
                        
            SelectInput(1, false);
            GetSelectedNeuronPos(pos.x, pos.y);
            ClearSelection();
            pos.Move(int16((inCount / 2) * (mHorSpacing + 1) * gridWidth) - 
                     int16((neuronCount / 2) * (mHorSpacing + 1)  * gridWidth), 0);
            newX = pos.x;
        }
        else
        {
            newX = uint16(MAX_COORDINATE / 2);
        }
        
        return newX;
    }
    
    // Get the Y-Position for a new hidden layer
    uint16 GetNewHidLayerYPos()
    {
        uint16 yPos;
        uint16 inCount = GetInputCount();
        uint16 contextCount = GetContextCount();
        
        if (inCount > 0)
        {
            if (contextCount > 0)
            {   // There is already a context layer (hidden neurons with connected 
                // outputs but unconnected inputs)
                // Get position of left most hidden neuron in context layer.
                Position pos;
                uint16 gridWidth = GetGridWidth();
                SelectContext(1, false);
                GetSelectedNeuronPos(pos.x, pos.y);
                ClearSelection();
                yPos = pos.y - (mVertSpacing + 1) * gridWidth;
            }
            else
            { // This is the first hidden layer. Place it above the output layer.
                int16 outCount = GetOutputCount();
                
                if (outCount > 0)
                {
                    // Get position of left most output neuron.
                    Position pos;
                    uint16 gridWidth = GetGridWidth();
                    SelectOutput(1, false);
                    GetSelectedNeuronPos(pos.x, pos.y);
                    ClearSelection();
                    yPos = pos.y - (mVertSpacing + 1) * gridWidth;
                }
                else
                {
                    yPos = uint16(MAX_COORDINATE / 2);
                }
            }
            
        }
        else
        {
            yPos = uint16(MAX_COORDINATE / 2);
        }
        
        return yPos;
    }
    
    // Normalize the input neuron with the given number
    void NormalizeInputNeuron(uint number, double actMin, double actMax)
    {
        SNeuronProp prop;
        
        SelectInput(number, false);
        GetSelectedNeuronProp(prop);
        prop.useNormalization = true;
        
        double delta = actMax - actMin;
        
        prop.normRangeLow = actMin - delta * INPUT_NORMALIZE_OVERHEAD_PERCENT / 100;;
        prop.normRangeHigh = actMax + delta * INPUT_NORMALIZE_OVERHEAD_PERCENT / 100;
        // We select the IDENTICAL 0 TO 1 function here for all inputs
        prop.actFunc = AF_IDENTICAL_0_1;
        SetSelectedNeuronProp(prop);
        ClearSelection();
    }
    
    // Normalize the output neuron with the given number
    void NormalizeOutputNeuron(uint number, double actMin, double actMax)
    {
        SNeuronProp prop;
        
        SelectOutput(number, false);
        GetSelectedNeuronProp(prop);
        prop.useNormalization = true;
        
        double delta = actMax - actMin;
        
        prop.normRangeLow = actMin - delta * OUTPUT_NORMALIZE_OVERHEAD_PERCENT / 100;
        prop.normRangeHigh = actMax + delta * OUTPUT_NORMALIZE_OVERHEAD_PERCENT / 100;
        SetSelectedNeuronProp(prop);
        ClearSelection();
    }
    
    // Get the input column number that corresponds to a given input neuron number
    uint GetLessonInputColumnNum(uint neuronNum)
    {
        string neuronName;
        uint columnNum = 0;
        
        if (GetInputName(neuronNum, neuronName))
        {
            string columnName;
            uint count = GetLessonInputCount();
            
            for (uint i = 1; i <= count; i++)
            {
                GetLessonInputName(i, columnName);
                if (neuronName == columnName)
                {
                    columnNum = i;
                    break;
                }
            }
        }
        
        return columnNum;
    }
    
    // Get the output column number that corresponds to a given input neuron number
    uint GetLessonOutputColumnNum(uint neuronNum)
    {
        string neuronName;
        uint columnNum = 0;
        
        if (GetOutputName(neuronNum, neuronName))
        {
            string columnName;
            uint count = GetLessonOutputCount();
            
            for (uint i = 1; i <= count; i++)
            {
                GetLessonOutputName(i, columnName);
                if (neuronName == columnName)
                {
                    columnNum = i;
                    break;
                }
            }
        }
        
        return columnNum;
    }
    
    // The origin position for editing
    Position mPos;
    // The horizontal spacing in units of the grid width
    uint16 mHorSpacing;
    // The vertical spacing in units of the grid width
    uint16 mVertSpacing;
};
