/*
This script performs an automated training and validation run of a MemBraion neural net:

This is what the script does:
- Load and randomize the neural net
- Load a training lesson from an mbl file
- Load a validation lesson from an mbl file
- Train the neural net for an adjustable time using the training lesson
- During the training visualize the net error for both training and validation lesson (inter-changing)
- During the training visualize the net output in reaction to both training and validation 
  lesson for all output neurons (inter-changing) 
- After the training export the validation output of the net into a csv file. The export contains the 
  validation inputs, the validation outputs of the net and the original validation target output data.

The net also prints information to the Script Trace window during execution

Several parameters can be adjusted in the beginning of the script.
*/

// This script uses timers. Always copy the script 'SecondsTimer.as' into the sub directory 'SecondsTimer'.
#include "SecondsTimer\SecondsTimer.as"


// --------------------------------- Adjustable constants section begin --------------------------------------

string sNetName = "Net1.mbn";                    // Name of the neural net
string sTrainLessonName = "Train.mbl";           // Name of the training lesson file
string sValidateLessonName = "Validate.mbl";     // Name of the validation lesson file
string sResultFileName = "ValidationResult.csv"; // Name of the validation result file name
string sTargetColumnExtension = "(Target)";		 // Column name extension string for target output values

// The overall training time [seconds]
uint TRAIN_TIME = 30;

// Number of seconds the training lesson shall be displayed in the net error window.
uint SHOW_TRAIN_LESSON_TIME = 4;

// Number of seconds the validation lesson shall be displayed in the net error window.
uint SHOW_VALIDATE_LESSON_TIME = 10;

// Number of sesonds after which the monitored output neuron shall be toggled
uint TOGGLE_MONITORED_OUTPUT_INTERVAL = 2;

// --------------------------------- Adjustable constants section end --------------------------------------

// Variable to detect and restore user setting: Capture Best Net on Stock during teaching?
// The setting is requested by the script in the beginning, then set to active (i.e. capturing) and reset
// to its user defined value again in the end of the script.
bool sCaptureBestNetOnStock = false;

// Program entry point
void main()
{
    Init();
    
    if (IsNetModified())
    {
    	EDlgRet ret = MessageBox("The currently open net is modified and has not been saved.\r\n"
    				   			 "This script will load a net from file and thus your changes will be lost.\r\n\r\n"
    				             "Continue script execution?", MB_YESNO); 
    	if (ret != IDYES)
    	{
    		AbortScript();
    	}
    }
    OpenNet(sNetName);
    RandomizeNet();
    
    SetLessonCount(2);
    
    // Load validation lesson to lesson #2
    SelectLesson(2);
    LoadLesson(sValidateLessonName);
    
    // Load train lesson to lesson #1
    SelectLesson(1);
    LoadLesson(sTrainLessonName);
    
    // Start with training lesson as the net error lesson.
    SelectNetErrLesson(2);
    
    SelectTeacher("RPROP");
    
    ShowErrorViewer(true);
    ShowPatternErrorViewer(true);
    
    // Perform the training section of the script
    ProcessTraining();
    
    // Load best net captured on stock and announce it to the user
    LoadBestNetFromStock();
    
    // Record the validation lesson to CSV
    RecordValidationLesson();
    
    Cleanup();
}

// Perform initialization
void Init()
{
	// Remember user setting and then activate capturing of the best net on stock
	sCaptureBestNetOnStock = GetCaptureBestNetOnStock();
	SetCaptureBestNetOnStock(true);
}


// Perform the training section of the script
void ProcessTraining()
{
    // Start the actual training
    StartTeaching();

	// Create the timer used to limit the overall training time
    SecondsTimer trainTimer(TRAIN_TIME);
    // Create the timer used to switch between training and validation lesson.
    SecondsTimer lessonTimer(SHOW_TRAIN_LESSON_TIME);
    // Create the timer used to toggle the monitored output neuron.
    SecondsTimer outputToggleTimer(TOGGLE_MONITORED_OUTPUT_INTERVAL);
        
    Trace("Display: Train Lesson\n");
    
    uint outputNeuron = 1;
    uint outputCount = GetOutputCount();
    bool showTrainLesson = true;
    
    // Show training lesson first in pattern error viewer
    SelectPatternErrorViewerLesson(TRAIN_LESSON);
	    
    while (!trainTimer.IsElapsed() && IsTeaching())
    {
        if (lessonTimer.IsElapsed())
        {
            if (showTrainLesson)
            { // Switch net error lesson to be the validation lesson
                lessonTimer.Start(SHOW_VALIDATE_LESSON_TIME);
                Trace("Display: Validation Lesson\n");
                showTrainLesson = false;
                SelectPatternErrorViewerLesson(NET_ERR_LESSON);
            }
            else
            { // Switch net error lesson to be the training lesson
                lessonTimer.Start(SHOW_TRAIN_LESSON_TIME);
                Trace("Display: Train Lesson\n");
                showTrainLesson = true;
                SelectPatternErrorViewerLesson(TRAIN_LESSON);
            }
        }
        
        if ((outputCount > 0) && outputToggleTimer.IsElapsed())
        { // It is time to toggle the monitored output neuron
            outputNeuron++;
            if (outputNeuron > outputCount)
            {
                outputNeuron = 1;
            }
            
            SelectPatternErrorViewerData(outputNeuron);
            outputToggleTimer.Start(TOGGLE_MONITORED_OUTPUT_INTERVAL);
        }
        
        // Leave some processing time for MemBrain itself.
        Sleep(100);
    }
    
    // Stop the teaching
    StopTeaching();
}


// Record the validation lesson and save it to file
void RecordValidationLesson()
{
    // Disable lesson editor update in order to improve performance
	EnableLessonEditorUpdate(false);
	
    SetLessonCount(3);
    SelectLesson(2);
 
    // Adjust the view
    ViewSetting(UPDATE_THINK, false);
    
    // Now prepare lesson #3 to record the output activation of the net
    SetRecordingType(RT_ACT);                // Record activation values of output neurons
    StartRecording(3);                       // Enable recording of data to lesson #2
    ShowLessonEditor(false);                 // Hide the lesson editor
    SetThinkSpeed(0);                        // Maximum think speed (0 ms wait between patterns)
    ThinkLesson();                           // Think on every pattern in lesson #1 (and record to #2)
    SleepExec();                             // Wait until 'Think on Lesson' is finished
    StopRecording();                         // Deactivate recording again
    
    // Append the original (target value) output columns to the recorded validation lesson
    AppendOutputTargetColumns(2, 3);
    
    // Now export the recorded lesson #3 to csv
    SelectLesson(3);
    ExportLessonRaw(sResultFileName);
    
    ViewSetting(UPDATE_THINK, true);// Re-enable update view on think steps
    string message = "The results from the net '" + sNetName + "' based on '" + sValidateLessonName + "' have been captured \n"
                     "in lesson #3 and also have been exported to '" + sResultFileName + "'";
                     
    MessageBox(message);
    
    EnableLessonEditorUpdate(true);	
}


// Append the original (target value) output columns to the recorded validation lesson
void AppendOutputTargetColumns(uint valiLesson, uint recordedLesson)
{
	SelectLesson(valiLesson);
	uint addedCount = GetLessonOutputCount();
	uint patternCount = GetLessonSize();
	SelectLesson(recordedLesson);
	uint oldOutCount = GetLessonOutputCount();
	SetLessonOutputCount(oldOutCount + addedCount);
	double act;
	string name;
	
	// Copy column names + name extention
	for (uint col = 1; col <= addedCount; col++)
	{  
		SelectLesson(valiLesson);
		GetLessonOutputName(col, name);
		SelectLesson(recordedLesson);
		SetLessonOutputName(oldOutCount + col, name + " " + sTargetColumnExtension);
	}
	
	// Copy data
	for (uint pat = 1; pat <= patternCount; pat++)
	{
		for (uint col = 1; col <= addedCount; col++)
		{  
			SelectLesson(valiLesson);
			SelectPattern(pat);
			GetPatternOutput(col, act);
			SelectLesson(recordedLesson);
			SelectPattern(pat);
			SetPatternOutput(oldOutCount + col, act);
		}
	}
}

// Load best net captured on stock and announce it to the user
void LoadBestNetFromStock()
{
	uint autoStockSlot = GetLastTeacherAutoCaptureStockSlot();
	
	if (autoStockSlot > 0)
	{
		string text = "The best net version during training (minimum validation error) has been\r\n"
					  "captured on the net stock in the slot " + autoStockSlot + "\r\n\r\n";
		text += "Shall the best net be re-loaded from stock now?\r\n\r\n";
		text += "Click <Yes> to load the best net from stock now or click <No> to go with the\r\n"
				"net version from the end of the training phase.";
		
		EDlgRet ret = MessageBox(text, MB_YESNO);
		
		if (ret == IDYES)
		{
			bool ok = LoadNetFromStock(autoStockSlot);
			// Also update the net error once we've loaded the best net
			EvaluateNetError();
			
			if (ok)
			{
				Trace("Best net loaded from stock slot " + autoStockSlot + ".\r\n");
			}
			else
			{
				Trace("Failed to laod net from stock slot " + autoStockSlot + ".\r\n");
			}
		}
		else
		{
			Trace("Loading best net from stock skipped by user.\r\n");
		}			   
	}
}


// Perform cleanup after script has ended
void Cleanup()
{
	// Restore user setting: Capturing of the best net on stock
	SetCaptureBestNetOnStock(sCaptureBestNetOnStock);
}


// Reserved function name: 
// Called by MemBrain in case the script is aborted
void OnAbortScript()
{
    StopTeaching();
    StopThink();
    ViewSetting(UPDATE_THINK, true);// Re-enable update view on think steps
    Cleanup();
}
