/*
    This script demonstrates how the script class 'NetEditor' can be used to 
    easily create neural nets.
    
    The script allows to create a net with an adjustable number of neurons in 
    each hidden layer either based on a MemBrain lesson (that defines the inputs 
    and outputs) or based on user input.
    When using a lesson then the script also performs the nomalization of the net 
    based on the minimum and maximum activation values contained in the lesson.
    
    The three core functions of the NetEditor class deomonstrated here are:
    
    NetEditor.CreateNet(...)
    NetEditor.CreateHiddenLayers(...)
    NetEditor.NormalizeNetWithActiveLesson()
*/

#include "NetEditor.as"

//---------------- 'Constants'------------------------------------

// Default number of input neurons for manual net creation
const int DEFAULT_INPUT_COUNT = 5;

// Default number of output neurons for manual net creation
const int DEFAULT_OUTPUT_COUNT = 3;

// Default hidden layer count
const int DEFAULT_HID_LAYER_COUNT = 2;

// Default neuron count for a hidden layer
const int DEFAULT_HID_LAYER_SIZE = 5;

// Horizontal neuron spacing in units of grid width
const uint16 HORIZ_SPACING = 1;

// Vertical  neuron spacing in units of grid width
const uint16 VERT_SPACING = 3;


//---------- global variables ------------------------------------

// Use a lesson to create the net?
bool gUseLesson = false;

// Number of input neurons to be created
int gInputCount = DEFAULT_INPUT_COUNT;

// Number of output neurons to be created
int gOutputCount = DEFAULT_OUTPUT_COUNT;

// The adjusted grid width (will be determined later on)
uint16 gGridWidth = 0;

// Maximum allowed neuron count per layer (will be determined later on)
int gMaxCountPerLayer = 0;

// Maximum number of allowed layers
int gMaxHidLayerCount = 10;

// This is the array that specifies the number of hidden layers to create 
// and the neuron count in each of these layers, starting from input towards 
// output. It is handed over to the net editor for creation of the hidden layers.
uint16[] gHidLayers(DEFAULT_HID_LAYER_COUNT);

//----------------------------------------------------------------

// The net editor object
NetEditor editor;


// Example that uses the timers
void main()
{
    // Initialize some script variables.
    Init();
    
    // Determine the number for hidden layers for the net
    DetermineHidLayerCount();
    
    // Create net based on lesson or manual input? 
    if (MessageBox("Do you want to create the net based on a MemBrain lesson file?", MB_YESNO) == IDYES)
    {
        gUseLesson = true;
        CreateIOFromMbl();
    }
    else
    {
        CreateIOFromUserInput();
    }
    
    // Determine the size for every hidden layer
    DetermineHidLayerSizes();
    // Now create the hidden layers
    editor.CreateHiddenLayers(gHidLayers);

    // Finally, randomize and save the net
    RandomizeNet();
    
    if (gUseLesson)
    {   
        if (!editor.NormalizeNetWithActiveLesson())
        {
            string text = "Error occurred during normalization!\n\nCheck plausibility of loaded lesson!\n"
                          "It must contain at least a minimum and a maximum value for every data column.";
            MessageBox(text);
        }
    }
    
    SaveNet("NewNet.mbn");
}



// Initialize derived variables
void Init()
{
    // The adjusted grid width
    gGridWidth = GetGridWidth();
    // Maximum allowed neuron count per layer
    gMaxCountPerLayer = MAX_COORDINATE / ((HORIZ_SPACING + 2) * gGridWidth);
        
    // Adjust the spacing of the editor
    editor.SetHorizontalSpacing(HORIZ_SPACING);
    editor.SetVerticalSpacing(VERT_SPACING);
}

// Determine how many normal hidden layers there shall be in the net
void DetermineHidLayerCount()
{
    int count;
    
    if (UserInput("Please enter the number of hidden layers for the net.", 
                  DEFAULT_HID_LAYER_COUNT, count) == IDOK)
    {
        if (count > gMaxHidLayerCount)
        {
            MessageBox("Too many hidden layers!");
            AbortScript();
        }
        // Success. Adjust the size of the hidden layer array now.
        gHidLayers.resize(count);
    }
    else
    {
        AbortScript();
    }
}

// Create the I/O neurons of the net using a MemBrain lesson file
void CreateIOFromMbl()
{
    string lessonFileName;
        
    if (FileOpenDlg("Select Lesson File", "mbl", "", lessonFileName) == IDOK)
    {
        // Load the lesson into #1 (delete all other lessons)
        SetLessonCount(1);
        LoadLesson(lessonFileName);
        gInputCount = GetLessonInputCount();
        gOutputCount = GetLessonOutputCount();
        if (gInputCount > gMaxCountPerLayer)
        {
            MessageBox("Too many input columns in lesson!");
            AbortScript();
        }
        else if (gOutputCount > gMaxCountPerLayer)
        {
            MessageBox("Too many output columns in lesson!");
            AbortScript();
        }
        
        // Now we tell the editor to actually create the input and output neurons.
        // The number of hidden layers is required here to leave space for them.
        editor.CreateNet(gInputCount, gOutputCount, gHidLayers.length());
        // Take I/O names from lesson
        NamesToNet();
        ShowLessonEditor(false);
    }
    else
    {
        AbortScript();
    }
}

// Create the I/O neurons of the net with user input
void CreateIOFromUserInput()
{
    if (UserInput("Please enter the number of input neurons.", DEFAULT_INPUT_COUNT, gInputCount) != IDOK)
    {
        AbortScript();
    }
    if (gInputCount > gMaxCountPerLayer)
    {
        MessageBox("Too many input neurons!");
        AbortScript();
    }
    if (UserInput("Please enter the number of output neurons.", DEFAULT_OUTPUT_COUNT, gOutputCount) != IDOK)
    {
        AbortScript();
    }
    if (gOutputCount > gMaxCountPerLayer)
    {
        MessageBox("Too many output neurons!");
        AbortScript();
    }
    
    editor.CreateNet(gInputCount, gOutputCount, gHidLayers.length());
}

// Determine the size for each of the hidden layers 
void DetermineHidLayerSizes()
{
    int count = gHidLayers.length();
    int layerSize = DEFAULT_HID_LAYER_SIZE;
    
    for (int i = 0; i < count; i++)
    {
        int layerNum = i + 1;
        
        if (UserInput("Please enter the number of neurons in the hidden layer # " + layerNum + ".", 
                      layerSize, layerSize) != IDOK)
        {
            AbortScript();
        }
        if (layerSize > gMaxCountPerLayer)
        {
            MessageBox("Too many neurons for one layer! (maximum is " + gMaxCountPerLayer + ")");
            AbortScript();
        }
        else
        {   // Everything fine. Store the layer size in the array
            gHidLayers[i] = layerSize;
        }
    }
}






