/*
Open this file with a text editor that supports C++ or Java syntax highlighting in order to best
display the contents.

The script will open and train several nets one after each other, determine minimum and final net
error values and summarize the training results in the scripting trace window as well as in a text file.
*/

// Number of neural nets to operate on
const uint8 NUMBER_OF_NETS = 5;

// Common part of the name of all used neural nets. To be extended by number of net 1 .. NUMBER_OF_NETS
const string NET_FILE_NAME_BASE = "MackeyGlass";

// Default training time per net [s]
const uint32 DEFAULT_TRAIN_TIME = 60;

// The training lesson file
const string TRAIN_LESSON_NAME = "MackeyGlassTrain.mbl";

// The validation lesson file
const string VALIDATE_LESSON_NAME = "MackeyGlassValidate.mbl";

// The teacher to be used
const string TEACHER_NAME = "RPROP";

// The extension appended to a net file name when being saved after training
const string NET_FILE_SAVE_AS_EXTENSION = "_Trained";

// File name for the summary
const string RESULT_SUMMARY_FILE_NAME = "TrainingResultSummary.txt";

// Field separator
const string LIST_SEPARATOR = ";";

// Decimal separator
const string DEC_SEPARATOR = ",";



// -------------------- variables with static storage duration ----------------------------------------

// The last known net error for every net
double[] sCurNetErr(NUMBER_OF_NETS);

// The last known minimum net error for every net
double[] sMinNetErr(NUMBER_OF_NETS);

// Index of current net
uint8	sCurNetIdx = 0;

// Training time in seconds
uint32	sTrainTime = 0;

// Has the summary file been opened successfully?
bool sSummaryFileOpen = false;

// The summary file
file sSummaryFile;





// This is the main script execution function
void main()
{
	StopThink();
	StopTeaching();
	Introduction();

	// Perform some initialization
	Init();

	// Open the first net, open all used windows and let the user arrange them
	OpenCurrentNet();
	OpenAllWindowsAndAnnounceArrangement();

	// Request some runtime parameters from the user
	RequestTrainingTime();

	// Train and validate all the nets
	for (uint8 i = 0; i < NUMBER_OF_NETS; i++)
	{
		TrainAndValidateNet(i);
	}

	SummarizeResults();
}


// Shows introduction text in message boxes
void Introduction()
{
	string msg =  	"This script mainly interacts with the user through messages printed in the scripting trace window.\n"
					"When user interaction is required the script will be suspended and a corresponding text message\n"
					"will appear in the bottom of the trace window.\n"
					"After completion of the user interaction press the 'Pause' button in the tool bar in order to release it\n"
					"or uncheck the main menu item <Scripting><Suspend Script Execution> to let the script continue.\n\n"
					"You can close the trace window during script execution if you like - it will re-appear automatically\n"
					"when needed.\n"
					"You can also close the other windows, they will remember their current locations when re-opened.\n";

	MessageBox(msg);

	msg = "Please note that this script will permanently change the target net error and the pattern selection method\n"
		  "of the teacher '" + TEACHER_NAME + "' !\n"
		  "Consider to re-adjust it according to your needs after the script has finished.\n\n"
		  "Continue script?";

	if (MessageBox(msg, MB_YESNO) != IDYES)
	{
		AbortScript();
	}
}




// Open all windows used during this script.
// Give the user the possibility to properly arrange all the open windows.
void OpenAllWindowsAndAnnounceArrangement()
{
	const string message = 	"Please arrange all open windows to best fit onto your screen. Note that there might be several overlapped windows.";

	ShowLessonEditor(false);
	ShowPatternErrorViewer(true);
	ShowErrorViewer(true);

	WaitForUser(message);
}


// Output a message in the trace window and suspend script to wait for user to continue script execution
void WaitForUser(const string &in message)
{
	// Open trace window for sure and leave one empty line after last text entry.
	ShowTraceWin(true);
	Trace("Script waiting for user interaction:\n\n");

	// Output the text and suspend script
	Trace(message + "\n\n");
	Trace("Release 'Pause' button in the toolbar when finished.\n\n");
	SuspendScript();
}

// Output a message in the trace window to be the next action of the script
void AnnounceAction(const string &in message)
{
	// Open trace window for sure and leave one empty line after last text entry.
	ShowTraceWin(true);
	Trace(message + "...\n");
}

// Get the file name for the given net index
string GetNetFileFileName(uint8 idx)
{
	return NET_FILE_NAME_BASE + (idx + 1) + ".mbn";
}

// Get the file name for the current net index
string GetCurrentNetFileName()
{
	return GetNetFileFileName(sCurNetIdx);
}

// Get the file name for the given net ndex to save the trained net as
string GetNetFileNameSaveAs(uint8 idx)
{
	return NET_FILE_NAME_BASE + (idx + 1) + NET_FILE_SAVE_AS_EXTENSION + ".mbn";
}

// Get the file name for the current net ndex to save the trained net as
string GetCurrentNetFileNameSaveAs()
{
	return GetNetFileNameSaveAs(sCurNetIdx);
}


// Open the net corresponding to the current net array index 'sCurNetIdx'
void OpenCurrentNet()
{
	OpenNet(GetCurrentNetFileName());
}



// Initialize net errors.
void Init()
{
	AnnounceAction("Initializing variables");
	// negative values for net errors indicate invalid values
	for (uint8 i = 0; i < NUMBER_OF_NETS; i++)
	{
		sCurNetErr[i] = -1.0;
		sMinNetErr[i] = -1.0;
	}

	sCurNetIdx = 0;

	// Load training and validation lesson, set training lesson to be the active one, set
	// validation lesson to be used for calculating the net error.
	AnnounceAction("Loading lessons and initializing Lesson Editor");
	SetLessonCount(2);
	SelectLesson(1);
	LoadLesson(TRAIN_LESSON_NAME);
	SelectLesson(2);
	LoadLesson(VALIDATE_LESSON_NAME);
	SelectLesson(1);
	SelectNetErrLesson(2);
	ShowLessonEditor(false);

	AnnounceAction("Selecting teacher");
	SelectTeacher(TEACHER_NAME);
	TeacherSetting(TARGET_ERR, 0.0);
	TeacherSetting(PATTERN_SELECT, ORDERED);
}


// Request the training time from the user
void RequestTrainingTime()
{
	EDlgRet ret = UserInput("Please enter training time per net in seconds.", DEFAULT_TRAIN_TIME, sTrainTime);

	if (ret != IDOK)
	{
		AbortScript();
	}
}

// Train and validate the net with the given 0-based index
void TrainAndValidateNet(uint8 netIdx)
{
	sCurNetIdx = netIdx;

	string action = "Training and validating net: ";

	action += GetCurrentNetFileName();
	AnnounceAction(action);

	OpenCurrentNet();
	RandomizeNet();
	StartTimer(sTrainTime * 1000);

	ETeachResult res = TR_OK;

	StartTeaching();

	// Teach as long as teacher signals OK and as long as timer is not expired.
	// Update net error values during teaching at regular intervals
	while ((res == TR_OK) && !IsTimerExpired() && IsTeaching())
	{
		Sleep(100);
		UpdateNetErrors();

		res = GetLastTeachResult();
	}

	if (IsTeaching())
	{
		// Stop teaching for sure
		StopTeaching();
	}

	SaveNet(GetCurrentNetFileNameSaveAs());
}


// Update the net error variables for later analysis
void UpdateNetErrors()
{
	double err = GetNetError();

	// Store current net error for the given net
	sCurNetErr[sCurNetIdx] = err;

	// Store minimum net error for the given net
	if ((sMinNetErr[sCurNetIdx] < 0) || (err < sMinNetErr[sCurNetIdx]))
	{
		sMinNetErr[sCurNetIdx] = err;
	}
}


// Provide a result summary of the training
void SummarizeResults()
{
	Trace("\n------------------ Training result summary start ----------------------\n\n");

	string result;

	// Try to open result summary file
	sSummaryFileOpen = sSummaryFile.Open(RESULT_SUMMARY_FILE_NAME, FILE_MODE_WRITE | FILE_MODE_CREATE | FILE_SHARE_DENY_NONE);

	if (!sSummaryFileOpen)
	{
		MessageBox("Unable to open summary file!");
	}

	// Create headline in summary file
	Append2SummaryFile("Net" + LIST_SEPARATOR + "Last Net Error" + LIST_SEPARATOR + "Min Net Error\n");

	double minNetErr = -1;
	uint8 bestNetIdx = 0;

	for (uint8 i = 0; i < NUMBER_OF_NETS; i++)
	{
		result = GetNetFileNameSaveAs(i);
		Trace(result + ":\n");
		Append2SummaryFile(result + LIST_SEPARATOR);

		result = sCurNetErr[i];
		result = ConvertDoubleStringToLocale(result);
		Trace("Last known net error: " + result + ":\n");
		Append2SummaryFile(result + LIST_SEPARATOR);

		result = sMinNetErr[i];
		result = ConvertDoubleStringToLocale(result);
		Trace("Minimum net error: " + result + ":\n\n");
		Append2SummaryFile(result + "\n");

		if ((minNetErr < 0) || (sCurNetErr[i] < minNetErr))
		{
			minNetErr = sCurNetErr[i];
			bestNetIdx = i;
		}
	}

	result = GetNetFileNameSaveAs(bestNetIdx);
	Trace("The best trained net is '" + result + "'\n");

	Trace("\n------------------ Training result summary end ----------------------\n\n");

	Append2SummaryFile("\nBest trained net: " + result + "\n");

	if (sSummaryFileOpen)
	{
		Trace("All results have been additionally summarized in the file '" + RESULT_SUMMARY_FILE_NAME + "'.\n");
		sSummaryFile.Close();
	}

}


// Append text to the summary file if open
void Append2SummaryFile(string text)
{
	if (sSummaryFileOpen)
	{
		if (!sSummaryFile.Write(text))
		{
			MessageBox("Error in writing to summary file!");
			sSummaryFile.Close();
			sSummaryFileOpen = false;
		}
	}
}


// Convert the given number string to incorporate local decimal separator
const string ConvertDoubleStringToLocale(const string &in numStr)
{
	if (DEC_SEPARATOR == ".")
	{ // Nothing to do if anglo-american notation is adjusted
		return numStr;
	}

	int idx = numStr.findFirst(".", 0);

	if (idx < 0)
	{ // Nothing to do if string does not contain separator
		return numStr;
	}

	// The string contains a separator. Replace by user defined separator now...

	// copy part left from separator
	string ret = numStr.substr(0, idx);

	// Add user defined separator
	ret += DEC_SEPARATOR;

	// copy part right from separator
	ret += numStr.substr(idx + 1, numStr.length() - idx - 1);

	return ret;
}
