﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;

namespace MemBrainLib
{
    public class NeuralNet : MB_DLL
    {
        /// <summary>
        /// Possible teach result return values
        /// </summary>
        public enum EMBTeachResult
        {
            MB_TR_OK,							///< Teach step OK, teacher not finished
            MB_TR_TARGET_NET_ERROR_REACHED,		///< The target error has been reached. Teacher finished.
            MB_TR_MAX_NEURONS_ADDED,			///< The maximum number of neurons has been added by the teacher. Teacher finished.
            MB_TR_TEACH_ABORTED,				///< The teacher has been aborted by another operation. Not used up to now
            MB_TR_NOT_IN_SYNC,					///< The net is not in sync with the lesson.
            MB_TR_WRONG_ACT_FUNCTION,			///< The net contains at least one neuron that has an activation function which is incompatible with the teacher
            MB_TR_OUT_OF_LESSON_RANGE,			///< The teacher operates outside the current lesson range. Not used up to now.
            MB_TR_ANALYSIS_ERROR,				///< Teaching is not possible because of architectural errors in net.
            MB_TR_LESSON_EMPTY,					///< Teaching is not possible because the currently active lesson is empty.
            MB_TR_NET_ERR_LESSON_EMPTY,         ///< Teaching is not possible because the currently active net error lesson is empty
            MB_TR_NET_ERR_NOT_IN_SYNC           ///< The net is not in sync with the net error lesson.
        };

        /// <summary>
        /// Possible activation functions
        /// </summary>
        public enum EMBActFunc
        {
            MB_AF_LOGISTIC         = 0,
            MB_AF_IDENTICAL        = 1,
            MB_AF_IDENTICAL_0_1    = 2,
            MB_AF_TAN_H            = 3,
            MB_AF_BINARY           = 4,
            MB_AF_MIN_EUCLID_DIST  = 5,
            MB_AF_IDENTICAL_M11    = 6,
            MB_AF_RELU             = 7,
            MB_AF_SOFTPLUS         = 8,
            MB_AF_BIN_DIFF         = 9
        };

        /// <summary>
        /// Possible input functions
        /// </summary>
        public enum EMBInputFunc
        {
            MB_IF_SUM = 0,
            MB_IF_MUL = 1
        };

        /// <summary>
        /// Possible output ranges
        /// </summary>
        public enum EMBOutputFireLevel
        {
            MB_OFL_1 = 0,	    // 0 or 1
            MB_OFL_ACT = 1  		// as activation function calculates
        } ;

        /// Parameter structure to describe neuron properties
        public struct SMBNeuronProp
        {
            public double act;
            public EMBInputFunc inputFunc;                  // see EMBInputFunc
            public EMBActFunc actFunc;                      // see EMBActFunc
            public double actThres;
            public bool lockActThres;
            public double actSustain;                       // 0..1
            public EMBOutputFireLevel outputFireLevel;      // see EMBOutputFireLevel
            public int outputRecovTime;                     // 1..100000
            public double fireThresLow;                     // fireThresHi must be >= fireThresLo
            public double fireThresHi;                      // fireThresHi must be >= fireThresLo
            public bool useNormalization;
            public double normRangeLow;
            public double normRangeHigh;
            public bool useActIgnoreVal;
            public double actIgnoreVal;
            public double expLogistic;
            public double parmTanHyp;
            public double leakage;
            public double binDiffSlope;
            public bool allowTeacherOutputConnect;
            public bool displayName;
            public bool displayAct;
            public bool isPixel;
            public int width;                               // 15 - 30000
        };

        /// Parameter structure to describe link properties
        public struct SMBLinkProp
        {
            public double weight;
            public bool lockWeight;
            public int length;             // 1..10000
            public bool displayWeight;
        };
        
        /// <summary>
        /// Create a new neural net and add it to the net management list.
        /// </summary>
        public NeuralNet()
        {
            // The first net always exists in the dll already and thus must not be added explicitly. 
            if (sFirstNetCreated)
            {
                MB_AddNet();
                DllIndex = MB_GetNetCount() - 1;
            }
            else
            {
                int netCount = MB_GetNetCount();

                if (netCount > 1)
                { // Something went wrong with the net count and our code. Delete all nets
                  // except for the first one.
                    for (int i = netCount - 1; i > 0; i--)
                    {
                        MB_SelectNet(i);
                        MB_DeleteNet(i);
                    }
                }
                // The first net has the index 0 in the dll
                DllIndex = 0;
                sFirstNetCreated = true;
            }

            // Ensure the new net is empy
            Clear();

            // Add the new net to the administration list
            Nets.Add(this);
        }

        /// <summary>
        /// The destructor must call a correction function for every existing neural net
        /// to cause it to update its Dll index if required.
        /// </summary>
        ~NeuralNet()
        {
            Nets.RemoveAt(DllIndex);
            MB_DeleteNet(DllIndex);
            foreach (NeuralNet net in Nets)
            {
                net.UpdateAfterNetDeletion(DllIndex);
            }
        }

        /// <summary>
        /// Get function for property 'DllIdx'
        /// </summary>
        public int DllIdx
        {
            get { return DllIndex; }
        }

        /// <summary>
        /// Update the class instance when another class instance with given Dll index has 
        /// been deleted.
        /// </summary>
        /// <param name="deletedIdx"></param>
        private void UpdateAfterNetDeletion(int deletedDllIndex)
        {
            // If a net with smaller Dll index than our own has been deleted we need to decrement
            // our own Dll index.
            if (deletedDllIndex < DllIndex)
            {
                DllIndex--;
            }
        }

        /// <summary>
        /// Prepare the dll for an operation on this specific net instance
        /// </summary>
        private void PrepareDllForThis()
        {
            bool ok = MB_SelectNet(DllIndex);

            Debug.Assert(ok, "Unable to prepare dll for net idx " + DllIndex.ToString());
        }

        
        /// <summary>
        /// The net's index within the Dll
        /// </summary>
        private int DllIndex;

        /// <summary>
        /// Has the first NN already been created?
        /// </summary>
        static private Boolean sFirstNetCreated = false;

        /// <summary>
        /// Array for the neural net instances
        /// </summary>
        static private ArrayList Nets = new ArrayList();



//####################### Methods for using the created nets ####################################

        /// <summary>
        /// Load the currently active neural net from the given *.mbn file (including path)
        /// </summary>
        /// <param name="fileName"></param>
        /// <returns></returns>
        public bool Load(string fileName)
        {
            PrepareDllForThis();
            return MB_LoadNet(fileName);
        }

        /// <summary>
        /// Save a MemBrain neural net to the given *.mbn file (including path)
        /// </summary>
        /// <param name="fileName"></param>
        /// <returns></returns>
        public bool SaveAs(string fileName)
        {
            PrepareDllForThis();
            return MB_SaveNetAs(fileName);
        }

        /// <summary>
        /// Save an already loaded MemBrain neural net (overwrite original file)
        /// </summary>
        /// <returns></returns>
        public bool Save()
        {
            PrepareDllForThis();
            return MB_SaveNet();
        }

        /// <summary>
        /// Reset the net. All activations and link spikes are set to 0.
        /// </summary>
        public void Reset()
        {
            PrepareDllForThis();
            MB_ResetNet();
        }

        /// <summary>
        /// Get number of input neurons in the net
        /// </summary>
        /// <returns></returns>
        public int GetInputCount()
        {
            PrepareDllForThis();
            return MB_GetInputCount();
        }

        /// <summary>
        /// Get number of output neurons in the net
        /// </summary>
        /// <returns></returns>
        public int GetOutputCount()
        {
            PrepareDllForThis();
            return MB_GetOutputCount();
        }

        /// <summary>
        /// Get name of input neuron at index idx.
        /// </summary>
        /// <param name="idx"></param>
        /// <returns>Name of input neuron or empty string if index is not valid</returns>
        public string GetInputName(int idx)
        {
            PrepareDllForThis();

            string name;

            if (MB_GetInputName(idx, out name))
            {
                return name.ToString();
            }
            else
            {
                return "";
            }
        }

        /// <summary>
        /// Get name of output neuron at index idx.
        /// </summary>
        /// <param name="idx"></param>
        /// <returns>Name of input neuron or empty string if index is not valid</returns>
        public string GetOutputName(int idx)
        {
            PrepareDllForThis();

            string name;

            if (MB_GetOutputName(idx, out name))
            {
                return name.ToString();
            }
            else
            {
                return "";
            }
        }

        /// <summary>
        /// Apply an activation value to the input neuron at index idx.
        /// </summary>
        /// <param name="idx"></param>
        /// <param name="act"></param>
        /// <returns></returns>
        public bool ApplyInputAct(int idx, double act)
        {
            PrepareDllForThis();
            return MB_ApplyInputAct(idx, act);
        }

        /// <summary>
        /// Get the activation value of the input neuron at index idx.
        /// </summary>
        /// <param name="idx"></param>
        /// <param name="pActVal"></param>
        /// <returns></returns>
        public bool GetInputAct(int idx, out double actVal)
        {
            PrepareDllForThis();
            return MB_GetInputAct(idx, out actVal);
        }

        /// <summary>
        /// Perform one think step with the net
        /// </summary>
        public void ThinkStep()
        {
            PrepareDllForThis();
            MB_ThinkStep();
        }

        /// <summary>
        /// Get the activation value of the output neuron at index idx.
        /// </summary>
        /// <param name="idx"></param>
        /// <param name="pActVal"></param>
        /// <returns></returns>
        public bool GetOutputAct(int idx, out double actVal)
        {
            PrepareDllForThis();
            return MB_GetOutputAct(idx, out actVal);
        }

        /// <summary>
        /// Get the output value of the output neuron at index idx.
        /// </summary>
        /// <param name="idx"></param>
        /// <param name="pActVal"></param>
        /// <returns></returns>
        public bool GetOutputOut(int idx, out double outVal)
        {
            PrepareDllForThis();
            return MB_GetOutputOut(idx, out outVal);
        }

        /// <summary>
        /// Get index of the last output winner neuron of the net. Return -1 if unknown. Else
        /// return the output neuron index of the winner neuron.
        /// </summary>
        /// <returns></returns>
        public int GetOutputWinnerNeuron()
        {
            PrepareDllForThis();
            return MB_GetOutputWinnerNeuron();
        }

        /// <summary>
        /// Get the activation range of the input neuron at index idx.
        /// </summary>
        /// <param name="idx"></param>
        /// <param name="actMinVal"></param>
        /// <param name="actMaxVal"></param>
        /// <returns></returns>
        public bool GetInputActRange(int idx, out double actMinVal, out double actMaxVal)
        {
            PrepareDllForThis();
            return MB_GetInputActRange(idx, out actMinVal, out actMaxVal);
        }

        /// <summary>
        /// Get the activation range of the output neuron at index idx
        /// </summary>
        /// <param name="idx"></param>
        /// <param name="actMinVal"></param>
        /// <param name="actMaxVal"></param>
        /// <returns></returns>
        public bool GetOutputActRange(int idx, out double actMinVal, out double actMaxVal)
        {
            PrepareDllForThis();
            return MB_GetOutputActRange(idx, out actMinVal, out actMaxVal);
        }

        /// <summary>
        /// Set the activation range of the input neuron at index idx
        /// </summary>
        /// <param name="idx"></param>
        /// <param name="actMinVal"></param>
        /// <param name="actMaxVal"></param>
        /// <returns></returns>
        public bool SetInputActRange(int idx, double actMinVal, double actMaxVal)
        {
            PrepareDllForThis();
            return MB_SetInputActRange(idx, actMinVal, actMaxVal);
        }

        /// <summary>
        /// Set the activation range of the output neuron at index idx
        /// </summary>
        /// <param name="idx"></param>
        /// <param name="actMinVal"></param>
        /// <param name="actMaxVal"></param>
        /// <returns></returns>
        public bool SetOutputActRange(int idx, double actMinVal, double actMaxVal)
        {
            PrepareDllForThis();
            return MB_SetOutputActRange(idx, actMinVal, actMaxVal);
        }

        /// <summary>
        /// Randomize the net
        /// </summary>
        public void Randomize()
        {
            PrepareDllForThis();
            MB_RandomizeNet();
        }

        /// Get the last known error of the net
        public double GetLastNetError()
        {
            PrepareDllForThis();
            return MB_GetLastNetError();
        }

        /// <summary>
        /// Perform a teach step with this net on the currently active lesson
        /// </summary>
        /// <returns></returns>
        public EMBTeachResult TeachStep()
        {
            PrepareDllForThis();
            return MB_TeachStep();
        }

        /// <summary>
        /// Conclude the current teach run. Should be called after every teach process completion.
        /// This function is static
        /// </summary>
        public static void StopTeaching()
        {
            MB_StopTeaching();
        }

        /// <summary>
        /// Set the password to be used when saving this net
        /// </summary>
        /// <param name="pwd">Password</param>
        /// <returns>true if successful</returns>
        public bool SetPassword(string pwd)
        {
            PrepareDllForThis();

            bool success = MB_SetNetFilePwd(pwd);

            return success;
        }

        /// <summary>
        /// Remove the password for this net. Will take effect when the net is saved to file.
        /// </summary>
        public void RemovePassword()
        {
            PrepareDllForThis();

            MB_RemoveNetFilePwd();
        }


        // -------------------- Functions for editing neural nets down from here ------------------------
        /// <summary>
        /// Clear selection
        /// </summary>
        public void ClearSelection()
        {
            PrepareDllForThis();
            MB_ClearSelection();
        }

        /// <summary>
        /// Select neuron(s) by their name.
        /// </summary>
        /// <returns>Number of found (and selected) neurons</returns>
        public int SelectNeuronsByName(string neuronName, bool addToSelection, bool findMultiple)
        {
            PrepareDllForThis();
            return MB_SelectNeuronsByName(neuronName, addToSelection, findMultiple);
        }

        /// <summary>
        /// Select an input neuron
        /// </summary>
        /// <param name="InNeuronIdx"></param>
        /// <param name="addToSelection"></param>
        /// <returns></returns>
        public bool SelectInput(int InNeuronIdx, bool addToSelection)
        {
            PrepareDllForThis();
            return MB_SelectInput(InNeuronIdx, addToSelection);
        }

        /// <summary>
        /// Select all input neurons
        /// </summary>
        /// <param name="addToSelection"></param>
        public void SelectInput(bool addToSelection)
        {
            PrepareDllForThis();
            MB_SelectAllInputs(addToSelection);
        }

        /// <summary>
        /// Select an output neuron
        /// </summary>
        /// <param name="outNeuronIdx"></param>
        /// <param name="addToSelection"></param>
        /// <returns></returns>
        public bool SelectOutput(int outNeuronIdx, bool addToSelection)
        {
            PrepareDllForThis();
            return MB_SelectOutput(outNeuronIdx, addToSelection);
        }

        /// <summary>
        /// Select all output neurons
        /// </summary>
        /// <param name="addToSelection"></param>
        public void SelectOutput(bool addToSelection)
        {
            PrepareDllForThis();
            MB_SelectAllOutputs(addToSelection);
        }

        /// <summary>
        /// Get the number of hidden layers in the net
        /// </summary>
        /// <returns></returns>
        public int GetHiddenLayerCount()
        {
            PrepareDllForThis();
            return MB_GetHiddenLayerCount();
        }

        /// <summary>
        /// Get the number of neurons in a given hidden layer
        /// </summary>
        /// <param name="hidLayerIdx"></param>
        /// <returns></returns>
        public int GetHiddenCount(int hidLayerIdx)
        {
            PrepareDllForThis();
            return MB_GetHiddenCount(hidLayerIdx);
        }

        /// <summary>
        /// Get the number of neurons in all hidden layers
        /// </summary>
        /// <returns></returns>
        public int MB_GetHiddenCount()
        {
            PrepareDllForThis();
            return MB_GetHiddenCountAll();
        }

        /// <summary>
        /// Get the number of neurons in the context layer
        /// </summary>
        /// <returns></returns>
        public int GetContextCount()
        {
            PrepareDllForThis();
            return MB_GetContextCount();
        }

        /// <summary>
        ///  Get the number of neurons in the unresolved layer
        /// </summary>
        /// <returns></returns>
        public int GetUnresolvedCount()
        {
            PrepareDllForThis();
            return MB_GetUnresolvedCount();
        }

        /// <summary>
        /// Select a specific hidden neuron
        /// </summary>
        /// <param name="hidLayerIdx"></param>
        /// <param name="hidNeuronIdx"></param>
        /// <param name="addToSelection"></param>
        /// <returns></returns>
        public bool SelectHidden(int hidLayerIdx, int hidNeuronIdx, bool addToSelection)
        {
            PrepareDllForThis();
            return MB_SelectHidden(hidLayerIdx, hidNeuronIdx, addToSelection);
        }

        /// <summary>
        /// Select all neurons in a hidden layer
        /// </summary>
        /// <param name="hidLayerIdx"></param>
        /// <param name="addToSelection"></param>
        /// <returns></returns>
        public bool SelectHidden(int hidLayerIdx, bool addToSelection)
        {
            PrepareDllForThis();
            return MB_SelectHiddenLayer(hidLayerIdx, addToSelection);
        }

        /// <summary>
        /// Select all hidden neurons
        /// </summary>
        /// <param name="addToSelection"></param>
        public void SelectHidden(bool addToSelection)
        {
            PrepareDllForThis();
            MB_SelectAllHidden(addToSelection);
        }

        /// <summary>
        /// Select a specific context neuron
        /// </summary>
        /// <param name="neuronIdx"></param>
        /// <param name="addToSelection"></param>
        /// <returns></returns>
        public bool SelectContext(int neuronIdx, bool addToSelection)
        {
            PrepareDllForThis();
            return MB_SelectContext(neuronIdx, addToSelection);
        }

        /// <summary>
        /// Select all context neurons
        /// </summary>
        /// <param name="addToSelection"></param>
        public void SelectContext(bool addToSelection)
        {
            PrepareDllForThis();
            MB_SelectAllContexts(addToSelection);
        }

        /// <summary>
        /// Select an unresolved neuron
        /// </summary>
        /// <param name="neuronIdx"></param>
        /// <param name="addToSelection"></param>
        /// <returns></returns>
        public bool SelectUnresolved(int neuronIdx, bool addToSelection)
        {
            PrepareDllForThis();
            return MB_SelectUnresolved(neuronIdx, addToSelection);
        }

        /// <summary>
        /// Select all unresolved neurons
        /// </summary>
        /// <param name="addToSelection"></param>
        public void SelectUnresolved(bool addToSelection)
        {
            PrepareDllForThis();
            MB_SelectAllUnresolved(addToSelection);
        }

        /// <summary>
        /// Clear the Extra Selection
        /// </summary>
        public void ClearExtraSelection()
        {
            PrepareDllForThis();
            MB_ClearExtraSelection();
        }

        /// <summary>
        /// Apply Extra Selection to the current selection
        /// </summary>
        public void ExtraSelection()
        {
            PrepareDllForThis();
            MB_ExtraSelection();
        }

        /// <summary>
        /// Connect FROM Extra Selection
        /// </summary>
        public void ConnectFromExtra()
        {
            PrepareDllForThis();
            MB_ConnectFromExtra();
        }

        /// <summary>
        /// Connect TO Extra Selection
        /// </summary>
        public void ConnectToExtra()
        {
            PrepareDllForThis();
            MB_ConnectToExtra();
        }

        /// <summary>
        /// Add an input neuron to the net
        /// </summary>
        /// <param name="posX"></param>
        /// <param name="posY"></param>
        /// <param name="name"></param>
        public void AddInput(int posX, int posY, string name)
        {
            PrepareDllForThis();
            MB_AddInput(posX, posY, name);
        }

        /// <summary>
        /// Add an output neuron to the net
        /// </summary>
        /// <param name="posX"></param>
        /// <param name="posY"></param>
        /// <param name="name"></param>
        public void AddOutput(int posX, int posY, string name)
        {
            PrepareDllForThis();
            MB_AddOutput(posX, posY, name);
        }

        /// <summary>
        /// Add an unresolved hidden neuron to the net
        /// </summary>
        /// <param name="posX"></param>
        /// <param name="posY"></param>
        /// <param name="name"></param>
        public void AddHidden(int posX, int posY, string name)
        {
            PrepareDllForThis();
            MB_AddHidden(posX, posY, name);
        }

        /// <summary>
        /// Get the properties of the currently selected neuron
        /// </summary>
        /// <param name="prop"></param>
        /// <returns></returns>
        public bool GetSelectedNeuronProp(out NeuralNet.SMBNeuronProp prop)
        {
            PrepareDllForThis();
            return MB_GetSelectedNeuronProp(out prop);
        }

        /// <summary>
        /// Set the properties of the selected neurons
        /// </summary>
        /// <param name="prop"></param>
        /// <returns></returns>
        public bool SetSelectedNeuronProp(ref NeuralNet.SMBNeuronProp prop)
        {
            PrepareDllForThis();
            return MB_SetSelectedNeuronProp(ref prop);
        }

        /// <summary>
        /// Delete the selected objects
        /// </summary>
        public void DeleteSelectedObjects()
        {
            PrepareDllForThis();
            MB_DeleteSelectedObjects();
        }

        /// <summary>
        /// Prepare for a new net (delete all content of the net)
        /// </summary>
        public void Clear()
        {
            PrepareDllForThis();
            MB_ClearNet();
        }

        /// <summary>
        /// Set the name of the selected neurons
        /// </summary>
        /// <param name="name"></param>
        public void SetSelectedNeuronName(string name)
        {
            PrepareDllForThis();
            MB_SetSelectedNeuronName(name);
        }

        /// <summary>
        /// Move all selected neurons
        /// </summary>
        /// <param name="dX"></param>
        /// <param name="dY"></param>
        public void MoveSelectedNeurons(int dX, int dY)
        {
            PrepareDllForThis();
            MB_MoveSelectedNeurons(dX, dY);
        }

        /// <summary>
        /// Select all links from Extra Selection to Selection
        /// </summary>
        public void SelectFromExtra()
        {
            PrepareDllForThis();
            MB_SelectFromExtra();
        }

        /// <summary>
        /// Select all links from Selection to Extra Selection
        /// </summary>
        public void SelectToExtra()
        {
            PrepareDllForThis();
            MB_SelectToExtra();
        }

        /// <summary>
        /// Get the properties of the currently selected link
        /// </summary>
        /// <param name="prop"></param>
        /// <returns></returns>
        public bool GetSelectedLinkProp(out NeuralNet.SMBLinkProp prop)
        {
            PrepareDllForThis();
            return MB_GetSelectedLinkProp(out prop);
        }

        /// <summary>
        /// Set the properties of the selected links
        /// </summary>
        /// <param name="prop"></param>
        /// <returns></returns>
        public bool SetSelectedLinkProp(ref NeuralNet.SMBLinkProp prop)
        {
            PrepareDllForThis();
            return MB_SetSelectedLinkProp(ref prop);
        }

        /// <summary>
        /// Get the position of the selected neuron
        /// </summary>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <returns></returns>
        public bool GetSelectedNeuronPos(out int x, out int y)
        {
            PrepareDllForThis();
            return MB_GetSelectedNeuronPos(out x, out y);
        }


    }
}
