﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Runtime.InteropServices;
/// Definition of the membrain integer as 32 bit integer
using MB_INT = System.Int32;


namespace MemBrainLib
{ 
    public class MB_DLL
    {
#if WIN64
        private const string MEMBRAIN_DLL_NAME = "MemBrainDLL64.dll";
#else
        private const string MEMBRAIN_DLL_NAME = "MemBrainDLL.dll";
#endif


        /// Possible return values of most MemBrain DLL interfaces
        protected enum EMBRetVal
        {
            MEMBRAIN_ERR = -1,      ///< An error occured
            MEMBRAIN_OK = 0,        ///< Call successful
        };

        /// Possible values for boolean integer parameter interpretation
        protected enum EMBBoolean
        {
            MEMBRAIN_TRUE = 1,     
            MEMBRAIN_FALSE = 0,    
        };

        /// Parameter structure to describe neuron properties
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        private struct SMBNeuronPropNative
        {
            public double act;
            public MB_INT inputFunc;                       // see EMBInputFunc
            public MB_INT actFunc;                         // see EMBActFunc
            public double actThres;
            public MB_INT lockActThres;
            public double actSustain;                      // 0..1
            public MB_INT outputFireLevel;                 // see EMBOutputFireLevel
            public MB_INT outputRecovTime;                 // 1..100000
            public double fireThresLow;                    // fireThresHi must be >= fireThresLo
            public double fireThresHi;                     // fireThresHi must be >= fireThresLo
            public MB_INT useNormalization;
            public double normRangeLow;
            public double normRangeHigh;
            public MB_INT useActIgnoreVal;
            public double actIgnoreVal;
            public double expLogistic;
            public double parmTanHyp;
            public double leakage;
            public double binDiffSlope;
            public MB_INT allowTeacherOutputConnect;
            public MB_INT displayName;
            public MB_INT displayAct;
            public MB_INT isPixel;
            public MB_INT width;                           // 15 - 30000
        };

   

        /// Parameter structure to describe link properties
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        private struct SMBLinkPropNative
        {
            public double weight;
            public MB_INT lockWeight;
            public MB_INT length;             // 1..10000
            public MB_INT displayWeight;
        } ;

        /// <summary>
        /// The maximum length of strings retrieved from the Dll
        /// </summary>
        protected const int MAX_STRING_LEN = 1000;

        /// Get the version information string of the DLL.
        /// Specify maximum length of string to be copied (excluding the terminating '\0'.
        /// A terminating '\0' will be attached in every case.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetVersionInfo(StringBuilder info, MB_INT maxLen);

        /// Add a new neural net to the DLL's internal array of neural nets.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_AddNet();

        /// Get number of currently available neural nets in the DLL's array
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetNetCount();

        /// Get index of the currently selected net
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetSelectedNet();

        /// Delete the neural net at index idx.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_DeleteNet(MB_INT idx);

        /// Select one of the available nets as the currently active one.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SelectNet(MB_INT idx);

        /// Load the currently active neural net from the given *.mbn file (including path)
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_LoadNet(string fileName);

        /// Save a MemBrain neural net to the given *.mbn file (including path)
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SaveNetAs(string fileName);

        /// Save an already loaded MemBrain neural net (overwrite original file)
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SaveNet();

        /// Reset the net. All activations and link spikes are set to 0.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_ResetNet();

        /// Get number of input neurons in the net
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetInputCount();

        /// Get number of output neurons in the net
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetOutputCount();

        /// Get name of input neuron at index idx. Specify maximum length
        /// of string to be copied (excluding the terminating '\0'.
        /// A terminating '\0' will be attached in every case.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetInputName(MB_INT idx, StringBuilder name, MB_INT maxLen);

        /// Get name of output neuron at index idx. Specify maximum length
        /// of string to be copied (excluding the terminating '\0'.
        /// A terminating '\0' will be attached in every case.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetOutputName(MB_INT idx, StringBuilder name, MB_INT maxLen);

        /// Apply an activation value to the input neuron at index <idx>.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_ApplyInputAct(MB_INT idx, double act);

        /// Get the activation value of the input neuron at index idx.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetInputAct(MB_INT idx, out double act);

        /// Perform one think step of the net
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_ThinkStep();

        /// Get the activation value of the output neuron at index idx.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetOutputAct(MB_INT idx, out double act);

        /// Get the output value of the output neuron at index idx.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetOutputOut(MB_INT idx, out double outVal);

        /// Get index of the last output winner neuron of the net. Return -1 if unknown. Else
        /// return the output neuron index of the winner neuron.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetOutputWinnerNeuron();

        /// Get the activation range of the input neuron at index idx.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetInputActRange(MB_INT idx, out double actMin, out double actMax);

        /// Get the activation range of the output neuron at index idx.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetOutputActRange(MB_INT idx, out double actMin, out double actMax);

        /// Set the activation range of the input neuron at index idx.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SetInputActRange(MB_INT idx, double actMin, double actMax);

        /// Set the activation range of the output neuron at index idx.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SetOutputActRange(MB_INT idx, double actMin, double actMax);

////--------------------- For lesson handling and teaching --------------------
        /// Load a lesson to be the currently active
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_LoadLesson(string filePath);

        /// Import the currently active lesson from csv
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_ImportLesson(string filePath);

        /// Import the currently active lesson from raw csv
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_ImportLessonRaw(string filePath);

        /// Import the currently active lesson inputs from raw csv
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_ImportLessonInputsRaw(string filePath);

        /// Import the currently active lesson outputs from raw csv
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_ImportLessonOutputsRaw(string filePath);

        /// Save the currently active lesson to its current file name
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SaveLesson();

        /// Save the currently active lesson to the given file name
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SaveLessonAs(string filePath);

        /// Export the currently active lesson to csv. Specify maxCols with 0 to export with the full width of all columns
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_ExportLesson(string filePath, MB_INT maxCols);

        /// Export the currently active lesson to raw csv. Specify maxCols with 0 to export with the full width of all columns
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_ExportLessonRaw(string filePath, MB_INT maxCols);

        /// Export the inputs of the currently active lesson to raw csv. Specify maxCols with 0 to export with the full width of all columns
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_ExportLessonInputsRaw(string filePath, MB_INT maxCols);

        /// Export the outputs of the currently active lesson to raw csv. Specify maxCols with 0 to export with the full width of all columns
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_ExportLessonOutputsRaw(string filePath, MB_INT maxCols);

        /// Set the number of inputs of the currently administered lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SetLessonInputCount(MB_INT count);

        /// Get the number of inputs of the currently administered lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetLessonInputCount();

        /// Set the number of outputs of the currently administered lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SetLessonOutputCount(MB_INT count);

        /// Get the number of outputs of the currently administered lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetLessonOutputCount();

        /// Set the input name at index idx of the currently active lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SetLessonInputName(MB_INT idx, string name);

        /// Get the input name at index idx of the currently active lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetLessonInputName(MB_INT idx, StringBuilder name, int maxLen);

        /// Set the output name at index idx of the currently active lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SetLessonOutputName(MB_INT idx, string pName);

        /// Get the output name at index idx of the currently active lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetLessonOutputName(MB_INT idx, StringBuilder name, int maxLen);

        /// Set the input value at index idx of the current pattern
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SetPatternInput(MB_INT idx, double value);

        /// Get the input value at index idx of the current pattern
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetPatternInput(MB_INT idx, out double value);

        /// Set the output value at index idx of the current pattern
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SetPatternOutput(MB_INT idx, double value);

        /// Get the output value at index idx of the current pattern
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetPatternOutput(MB_INT idx, out double value);

        /// Select the currently active pattern of the currently active lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SelectPattern(MB_INT idx);

        /// Get the currently selected pattern index of the currently active lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetSelectedPattern();

        /// Delete the currently active pattern of the currently active lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_DeletePattern();

        /// Add a pattern to the end of the active lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_AddPattern();

        /// Get the number of patterns in the active lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetLessonSize();

        /// Enable/Disable the output data section of the active lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_EnableLessonOutData(MB_INT outDataEnabled);

        /// Transfer I/O names names and count from the currently active net to
        /// the currently active lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_NamesFromNet();

        /// Transfer I/O names names from the currently active lesson to
        /// the currently active net
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_NamesToNet();

        /// Set the number of currently administered lessons
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SetLessonCount(int count);

        /// Get the number of currently administered lessons
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetLessonCount();

        /// Select the active lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SelectLesson(MB_INT idx);

        /// Get the index of the currently active lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetSelectedLesson();

        // Select the net error lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SelectNetErrLesson(MB_INT idx);

        /// Get the index of the currently active net error lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetSelectedNetErrLesson();

        /// Apply the currently active pattern to the inputs of the currently active net
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_ApplyPattern();

        /// Select the current recording type: 0 = Activation, 1 = Output
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SetRecordingType(MB_INT type);

        /// Start recording data to a lesson: Specify lesson index to record to and step count
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_StartRecording(MB_INT lessonIdx, MB_INT stepCount);

        /// Stop recording data to lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_StopRecording();

        /// Think on all patters of the currently active lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_ThinkLesson();

        /// Load a teacher file
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_LoadTeacherFile(string filePath);

        /// Select the active teacher by name
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SelectTeacher(string name);

        /// Perform one teach step (lesson run). Return result according to Teacher.h (TR_....)
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern NeuralNet.EMBTeachResult _MB_TeachStep();

        /// Conclude the current teach run. Should be called after every teach process completion.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_StopTeaching();

        /// Get the number of available teachers
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetTeacherCount();

        /// Get the name of the teacher at index idx Specify maximum length
        /// of string to be copied (excluding the terminating '\0'.
        /// A terminating '\0' will be attached in every case.
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetTeacherName(MB_INT idx, StringBuilder name, MB_INT maxLen);

        /// Randomize the currently active net
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_RandomizeNet();

        /// Get the last known error of the currently active net
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern double _MB_GetLastNetError();

        /// Create an FFT lesson from the currently active lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_CreateFftLesson(bool complex, bool inputsAreColumns, MB_INT minFreqIdx, MB_INT maxFreqPoints);

        /// Get freqeuncy that corresponds to a frequency index in an FFT
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern double _MB_GetFftFrequency(MB_INT freqIdx, double overallSampleTime);

        /// Create a new lesson with averaged inputs from the current lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_CreateAverageLesson(MB_INT newInputDimension);

        /// Get the separators used for CSV file parsing
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetCsvFileSeparators(StringBuilder listSep, StringBuilder decSep);

        /// Set the separators used for CSV file parsing
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SetCsvFileSeparators(string listSep, string decSep);

// -------------------- Functions for editing neural nets down from here ------------------------
        /// Clear selection
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_ClearSelection();

        /// Select neurons by name
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_SelectNeuronsByName(string neuronName, bool addToSelection, bool findMultiple);

        /// Select an input neuron
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SelectInput(MB_INT InNeuronIdx, bool addToSelection);

        /// Select all input neurons
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_SelectAllInputs(bool addToSelection);

        /// Select an output neuron
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SelectOutput(MB_INT outNeuronIdx, bool addToSelection);

        /// Select all output neurons
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_SelectAllOutputs(bool addToSelection);

        /// Get the number of hidden layers in the net
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetHiddenLayerCount();            

        /// Get the number of neurons in a given hidden layer
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetHiddenCount(MB_INT hidLayerIdx);  

        /// Get the number of neurons in all hidden layers
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetHiddenCountAll();  

        /// Get the number of neurons in the context layer
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetContextCount();   

        /// Get the number of neurons in the unresolved layer
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetUnresolvedCount(); 

        /// Select a hidden neuron
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SelectHidden(MB_INT hidLayerIdx, MB_INT hidNeuronIdx, bool addToSelection);

        /// Select all neurons in a hidden layer
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SelectHiddenLayer(MB_INT hidLayerIdx, bool addToSelection);

        /// Select all hidden neurons
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_SelectAllHidden(bool addToSelection);

        /// Select a context neuron
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SelectContext(MB_INT neuronIdx, bool addToSelection);

        /// Select all context neurons
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_SelectAllContexts(bool addToSelection);

        /// Select an unresolved neuron
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SelectUnresolved(MB_INT neuronIdx, bool addToSelection);

        /// Select all unresolved neurons
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_SelectAllUnresolved(bool addToSelection);

        /// Clear the Extra Selection
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_ClearExtraSelection();

        /// Apply Extra Selection to the current selection
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_ExtraSelection();

        /// Connect FROM Extra Selection
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_ConnectFromExtra();  

        /// Connect TO Extra Selection
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_ConnectToExtra();  

        /// Add an input neuron to the net
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_AddInput(MB_INT posX, MB_INT posY, string name); 

        /// Add an output neuron to the net
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_AddOutput(MB_INT posX, MB_INT posY, string name); 

        /// Add an unresolved hidden neuron to the net
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_AddHidden(MB_INT posX, MB_INT posY, string name); 


        /// Get the properties of the currently selected neuron
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetSelectedNeuronProp(out SMBNeuronPropNative prop);

        /// Set the properties of the selected neurons
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SetSelectedNeuronProp(ref SMBNeuronPropNative prop);

        /// Delete the selected objects
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_DeleteSelectedObjects();

        /// Prepare for a new net
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_ClearNet();

        /// Set the name of the selected neurons    
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_SetSelectedNeuronName(string name);

        /// Move all selected neurons
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_MoveSelectedNeurons(MB_INT dX, MB_INT dY);

        /// Select all links from Extra Selection to Selection
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_SelectFromExtra();

        /// Select all links from Selection to Extra Selection
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_SelectToExtra();  


        /// Get the properties of the selected link
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetSelectedLinkProp(out SMBLinkPropNative prop);

        /// Set the properties of the selected links
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SetSelectedLinkProp(ref SMBLinkPropNative prop);

        /// Get the position of the selected neuron
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetSelectedNeuronPos(out MB_INT x, out MB_INT y);          

        /// Get the nearest grid point to a given point
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_GetGridPoint(MB_INT x, MB_INT y, out MB_INT gridX, out MB_INT gridY);

        /// Get the adjusted grid width
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern MB_INT _MB_GetGridWidth();      

        /// Get a random value between 0 an 1
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern double _MB_Random(); 

        /// Set the password for the currently loaded net
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SetNetFilePwd(string pwd);

        /// Remove the password for the currently loaded net
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_RemoveNetFilePwd();

        /// Set the password for the currently selected lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SetLessonFilePwd(string pwd);

        /// Remove the password for the currently selected lesson
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_RemoveLessonFilePwd();

        /// Set the password used for opening files
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern EMBRetVal _MB_SetFileOpenPwd(string pwd);

        /// Remove the password used for opening files
        [DllImport(MEMBRAIN_DLL_NAME)]
        static extern void _MB_RemoveFileOpenPwd();

      

// =============================================================================================================================
// interface functions to c# program
// =============================================================================================================================

        /// Get the version information string of the DLL.
        static protected bool MB_GetVersionInfo(out string verStr)
        {
            StringBuilder builder = new StringBuilder(MAX_STRING_LEN + 1);
            
            bool ok = (_MB_GetVersionInfo(builder, MAX_STRING_LEN) == EMBRetVal.MEMBRAIN_OK);

            if (ok)
            {
                verStr = builder.ToString();
            }
            else
            {
                verStr = "";
            }
            
            return (ok);
        }

        /// Add a new neural net to the DLL's internal array of neural nets.
        static protected void MB_AddNet()
        {
            _MB_AddNet();
        }

        /// Get number of currently available neural nets in the DLL's array
        static protected int MB_GetNetCount()
        {
            return _MB_GetNetCount();
        }

        /// Get index of the currently selected net
        static protected int MB_GetSelectedNet()
        {
            return _MB_GetSelectedNet();
        }

        /// Delete the neural net at index idx.
        static protected bool MB_DeleteNet(int idx)
        {
            return (_MB_DeleteNet(idx) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Select one of the available nets as the currently active one.
        static protected bool MB_SelectNet(int idx)
        {
            return (_MB_SelectNet(idx) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Load the currently active neural net from the given *.mbn file (including path)
        static protected bool MB_LoadNet(string fileName)
        {
            return (_MB_LoadNet(fileName) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Save a MemBrain neural net to the given *.mbn file (including path)
        static protected bool MB_SaveNetAs(string fileName)
        {
            return (_MB_SaveNetAs(fileName) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Save an already loaded MemBrain neural net (overwrite original file)
        static protected bool MB_SaveNet()
        {
            return (_MB_SaveNet() == EMBRetVal.MEMBRAIN_OK);
        }

        /// Reset the net. All activations and link spikes are set to 0.
        static protected void MB_ResetNet()
        {
            _MB_ResetNet();
        }

        /// Get number of input neurons in the net
        static protected int MB_GetInputCount()
        {
            return _MB_GetInputCount();
        }

        /// Get number of output neurons in the net
        static protected int MB_GetOutputCount()
        {
            return _MB_GetOutputCount();
        }

        /// Get name of input neuron at index idx. 
        static protected bool MB_GetInputName(int idx, out string name)
        {
            StringBuilder builder = new StringBuilder(MAX_STRING_LEN + 1);

            bool ok = (_MB_GetInputName(idx, builder, MAX_STRING_LEN) == EMBRetVal.MEMBRAIN_OK);

            if (ok)
            {
                name = builder.ToString();
            }
            else
            {
                name = "";
            }

            return (ok);
        }

        /// Get name of output neuron at index idx. 
        static protected bool MB_GetOutputName(int idx, out string name)
        {
            StringBuilder builder = new StringBuilder(MAX_STRING_LEN + 1);

            bool ok = (_MB_GetOutputName(idx, builder, MAX_STRING_LEN) == EMBRetVal.MEMBRAIN_OK);

            if (ok)
            {
                name = builder.ToString();
            }
            else
            {
                name = "";
            }

            return (ok);
        }

        /// Apply an activation value to the input neuron at index idx.
        static protected bool MB_ApplyInputAct(MB_INT idx, double act)
        {
            return (_MB_ApplyInputAct(idx, act) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Get the activation value of the input neuron at index idx.
        static protected bool MB_GetInputAct(int idx, out double actVal)
        {
            return (_MB_GetInputAct(idx, out actVal) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Perform one think step of the net
        static protected void MB_ThinkStep()
        {
            _MB_ThinkStep();
        }

        /// Get the activation value of the output neuron at index idx.
        static protected bool MB_GetOutputAct(int idx, out double actVal)
        {
            return (_MB_GetOutputAct(idx, out actVal) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Get the output value of the output neuron at index idx.
        static protected bool MB_GetOutputOut(int idx, out double outVal)
        {
            return (_MB_GetOutputOut(idx, out outVal) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Get index of the last output winner neuron of the net. Return -1 if unknown. Else
        /// return the output neuron index of the winner neuron.
        static protected int MB_GetOutputWinnerNeuron()
        {
            return _MB_GetOutputWinnerNeuron();
        }

        /// Get the activation range of the input neuron at index idx.
        static protected bool MB_GetInputActRange(int idx, out double actMinVal, out double actMaxVal)
        {
            return (_MB_GetInputActRange(idx, out actMinVal, out actMaxVal) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Get the activation range of the output neuron at index idx.
        static protected bool MB_GetOutputActRange(MB_INT idx, out double actMinVal, out double actMaxVal)
        {
            return (_MB_GetOutputActRange(idx, out actMinVal, out actMaxVal) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Set the activation range of the input neuron at index idx.
        static protected bool MB_SetInputActRange(int idx, double actMinVal, double actMaxVal)
        {
            return (_MB_SetInputActRange(idx, actMinVal, actMaxVal) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Set the activation range of the output neuron at index idx.
        static protected bool MB_SetOutputActRange(MB_INT idx, double actMinVal, double actMaxVal)
        {
            return (_MB_SetOutputActRange(idx, actMinVal, actMaxVal) == EMBRetVal.MEMBRAIN_OK);
        }

        ////--------------------- For lesson handling and teaching --------------------

        /// Load a lesson to be the currently active
        static protected bool MB_LoadLesson(string filePath)
        {
            return (_MB_LoadLesson(filePath) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Import the currently active lesson from csv
        static protected bool MB_ImportLesson(string filePath)
        {
            return (_MB_ImportLesson(filePath) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Import the currently active lesson from raw csv
        static protected bool MB_ImportLessonRaw(string filePath)
        {
            return (_MB_ImportLessonRaw(filePath) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Import the currently active lesson inputs from raw csv
        static protected bool MB_ImportLessonInputsRaw(string filePath)
        {
            return (_MB_ImportLessonInputsRaw(filePath) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Import the currently active lesson outputs from raw csv
        static protected bool MB_ImportLessonOutputsRaw(string filePath)
        {
            return (_MB_ImportLessonOutputsRaw(filePath) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Save the currently active lesson to its current file name
        static protected bool MB_SaveLesson()
        {
            return (_MB_SaveLesson() == EMBRetVal.MEMBRAIN_OK);
        }

        /// Save the currently active lesson to the given file name
        static protected bool MB_SaveLessonAs(string filePath)
        {
            return (_MB_SaveLessonAs(filePath) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Export the currently active lesson to csv. Specify maxCols with 0 to export with the full width of all columns
        static protected bool MB_ExportLesson(string filePath, int maxCols)
        {
            return (_MB_ExportLesson(filePath, maxCols) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Export the currently active lesson to raw csv. Specify maxCols with 0 to export with the full width of all columns
        static protected bool MB_ExportLessonRaw(string filePath, int maxCols)
        {
            return (_MB_ExportLessonRaw(filePath, maxCols) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Export the inputs of the currently active lesson to raw csv. Specify maxCols with 0 to export with the full width of all columns
        static protected bool MB_ExportLessonInputsRaw(string filePath, int maxCols)
        {
            return (_MB_ExportLessonInputsRaw(filePath, maxCols) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Export the outputs of the currently active lesson to raw csv. Specify maxCols with 0 to export with the full width of all columns
        static protected bool MB_ExportLessonOutputsRaw(string filePath, int maxCols)
        {
            return (_MB_ExportLessonOutputsRaw(filePath, maxCols) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Set the number of inputs of the currently administered lesson
        static protected bool MB_SetLessonInputCount(int count)
        {
            return (_MB_SetLessonInputCount(count) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Get the number of inputs of the currently administered lesson
        static protected int MB_GetLessonInputCount()
        {
            return _MB_GetLessonInputCount();
        }

        /// Set the number of outputs of the currently administered lesson
        static protected bool MB_SetLessonOutputCount(int count)
        {
            return (_MB_SetLessonOutputCount(count) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Get the number of outputs of the currently administered lesson
        static protected int MB_GetLessonOutputCount()
        {
            return _MB_GetLessonOutputCount();
        }

        /// Set the input name at index idx of the currently active lesson
        static protected bool MB_SetLessonInputName(int idx, string pName)
        {
            return (_MB_SetLessonInputName(idx, pName) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Get the input name at index idx of the currently active lesson
        static protected bool MB_GetLessonInputName(int idx, out string name)
        {
            StringBuilder builder = new StringBuilder(MAX_STRING_LEN + 1);

            bool ok = (_MB_GetLessonInputName(idx, builder, MAX_STRING_LEN) == EMBRetVal.MEMBRAIN_OK);

            if (ok)
            {
                name = builder.ToString();
            }
            else
            {
                name = "";
            }

            return (ok);
        }

        /// Set the output name at index idx of the currently active lesson
        static protected bool MB_SetLessonOutputName(int idx, string pName)
        {
            return (_MB_SetLessonOutputName(idx, pName) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Get the output name at index idx of the currently active lesson
        static protected bool MB_GetLessonOutputName(int idx, out string name)
        {
            StringBuilder builder = new StringBuilder(MAX_STRING_LEN + 1);

            bool ok = (_MB_GetLessonOutputName(idx, builder, MAX_STRING_LEN) == EMBRetVal.MEMBRAIN_OK);

            if (ok)
            {
                name = builder.ToString();
            }
            else
            {
                name = "";
            }

            return (ok);
        }

        /// Set the input value at index idx of the current pattern
        static protected bool MB_SetPatternInput(int idx, double value)
        {
            return (_MB_SetPatternInput(idx, value) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Get the input value at index idx of the current pattern
        static protected bool MB_GetPatternInput(int idx, out double value)
        {
            return (_MB_GetPatternInput(idx, out value) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Set the output value at index idx of the current pattern
        static protected bool MB_SetPatternOutput(int idx, double value)
        {
            return (_MB_SetPatternOutput(idx, value) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Get the output value at index idx of the current pattern
        static protected bool MB_GetPatternOutput(int idx, out double value)
        {
            return (_MB_GetPatternOutput(idx, out value) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Select the currently active pattern of the currently active lesson
        static protected bool MB_SelectPattern(int idx)
        {
            return (_MB_SelectPattern(idx) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Get the currently selected pattern index of the currently active lesson
        static protected int MB_GetSelectedPattern()
        {
            return _MB_GetSelectedPattern();
        }

        /// Delete the currently active pattern of the currently active lesson
        static protected bool MB_DeletePattern()
        {
            return (_MB_DeletePattern() == EMBRetVal.MEMBRAIN_OK);
        }

        /// Add a pattern to the end of the active lesson
        static protected void MB_AddPattern()
        {
            _MB_AddPattern();
        }

        /// Get the number of patterns in the active lesson
        static protected int MB_GetLessonSize()
        {
            return _MB_GetLessonSize();
        }

        /// Enable/Disable the output data section of the active lesson
        static protected bool MB_EnableLessonOutData(int outDataEnabled)
        {
            return (_MB_EnableLessonOutData(outDataEnabled) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Transfer I/O names names and count from the currently active net to
        /// the currently active lesson
        static protected bool MB_NamesFromNet()
        {
            return (_MB_NamesFromNet() == EMBRetVal.MEMBRAIN_OK);
        }

        /// Transfer I/O names names from the currently active lesson to
        /// the currently active net
        static protected bool MB_NamesToNet()
        {
            return (_MB_NamesToNet() == EMBRetVal.MEMBRAIN_OK);
        }

        /// Set the number of currently administered lessons
        static protected bool MB_SetLessonCount(int count)
        {
            return (_MB_SetLessonCount(count) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Get the number of currently administered lessons
        static protected int MB_GetLessonCount()
        {
            return _MB_GetLessonCount();
        }

        /// Select the active lesson
        static protected bool MB_SelectLesson(int idx)
        {
            return (_MB_SelectLesson(idx) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Get the index of the currently active lesson
        static protected int MB_GetSelectedLesson()
        {
            return _MB_GetSelectedLesson();
        }

        // Select the net error lesson
        static protected bool MB_SelectNetErrLesson(int idx)
        {
            return (_MB_SelectNetErrLesson(idx) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Get the index of the currently active net error lesson
        static protected int MB_GetSelectedNetErrLesson()
        {
            return _MB_GetSelectedNetErrLesson();
        }

        /// Apply the currently active pattern to the inputs of the currently active net
        static protected bool MB_ApplyPattern()
        {
            return (_MB_ApplyPattern() == EMBRetVal.MEMBRAIN_OK);
        }

        /// Select the current recording type: 0 = Activation, 1 = Output
        static protected bool MB_SetRecordingType(int type)
        {
            return (_MB_SetRecordingType(type) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Start recording data to a lesson: Specify lesson index to record to and step count
        static protected bool MB_StartRecording(int lessonIdx, int stepCount)
        {
            return (_MB_StartRecording(lessonIdx, stepCount) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Stop recording data to lesson
        static protected void MB_StopRecording()
        {
            _MB_StopRecording();
        }

        /// Think on all patters of the currently active lesson
        static protected bool MB_ThinkLesson()
        {
            return (_MB_ThinkLesson() == EMBRetVal.MEMBRAIN_OK);
        }

        /// Load a teacher file
        static protected bool MB_LoadTeacherFile(string filePath)
        {
            return (_MB_LoadTeacherFile(filePath) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Select the active teacher by name
        static protected bool MB_SelectTeacher(string name)
        {
            return (_MB_SelectTeacher(name) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Perform one teach step (lesson run). Return result according to Teacher.h (TR_....)
        static protected NeuralNet.EMBTeachResult MB_TeachStep()
        {
            return _MB_TeachStep();
        }

        /// Conclude the current teach run. Should be called after every teach process completion.
        static protected void MB_StopTeaching()
        {
            _MB_StopTeaching();
        }

        /// Get the number of available teachers
        static protected int MB_GetTeacherCount()
        {
            return _MB_GetTeacherCount();
        }

        /// Get the name of the teacher at index idx Specify maximum length
        /// of string to be copied (excluding the terminating '\0'.
        /// A terminating '\0' will be attached in every case.
        static protected bool MB_GetTeacherName(int idx, out string name)
        {
            StringBuilder builder = new StringBuilder(MAX_STRING_LEN + 1);

            bool ok = (_MB_GetTeacherName(idx, builder, MAX_STRING_LEN) == EMBRetVal.MEMBRAIN_OK);

            if (ok)
            {
                name = builder.ToString();
            }
            else
            {
                name = "";
            }

            return (ok);
        }

        /// Randomize the currently active net
        static protected void MB_RandomizeNet()
        {
            _MB_RandomizeNet();
        }

        /// Get the last known error of the currently active net
        static protected double MB_GetLastNetError()
        {
            return _MB_GetLastNetError();
        }

        /// Create an FFT lesson from the currently active lesson
        static protected bool MB_CreateFftLesson(bool complex, bool inputsAreColumns, int minFreqIdx, int maxFreqPoints)
        {
            bool ok = (_MB_CreateFftLesson(complex, inputsAreColumns, minFreqIdx, maxFreqPoints) == EMBRetVal.MEMBRAIN_OK);

            return ok;
        }

        /// Get freqeuncy that corresponds to a frequency index in an FFT
        static protected double MB_GetFftFrequency(int freqIdx, double overallSampleTime)
        {
            return _MB_GetFftFrequency(freqIdx, overallSampleTime);
        }

        /// Create a new lesson with averaged inputs from the current lesson
        static protected bool MB_CreateAverageLesson(int newInputDimension)
        {
            bool ok = (_MB_CreateAverageLesson(newInputDimension) == EMBRetVal.MEMBRAIN_OK);

            return ok;
        }

        /// Get the separators used for CSV file parsing
        static protected bool MB_GetCsvFileSeparators(out string listSep, out string decSep)
        {
            StringBuilder listSepBuilder = new StringBuilder(1 + 1);
            StringBuilder decSepBuilder = new StringBuilder(1 + 1);

            bool ok = (_MB_GetCsvFileSeparators(listSepBuilder, decSepBuilder) == EMBRetVal.MEMBRAIN_OK);

            if (ok)
            {
                listSep = listSepBuilder.ToString();
                decSep = decSepBuilder.ToString();
            }
            else
            {
                listSep = "";
                decSep = "";
            }

            return (ok);
        }

        /// Set the separators used for CSV file parsing
        static protected bool MB_SetCsvFileSeparators(string listSep, string decSep)
        {
            bool ok = (_MB_SetCsvFileSeparators(listSep, decSep) == EMBRetVal.MEMBRAIN_OK);

            return (ok);
        }
		
		/// Set the password for the currently loaded net
        static protected bool MB_SetNetFilePwd(string pwd)
        {
            return _MB_SetNetFilePwd(pwd) == EMBRetVal.MEMBRAIN_OK;
        }

        /// Remove the password for the currently loaded net
        static protected void MB_RemoveNetFilePwd()
        {
            _MB_RemoveNetFilePwd();
        }

        /// Set the password for the currently selected lesson
        static protected bool MB_SetLessonFilePwd(string pwd)
        {
            return _MB_SetLessonFilePwd(pwd) == EMBRetVal.MEMBRAIN_OK;
        }

        /// Remove the password for the currently selected lesson
        static protected void MB_RemoveLessonFilePwd()
        {
            _MB_RemoveLessonFilePwd();
        }

        /// Set the password for opening files
        static protected bool MB_SetFileOpenPwd(string pwd)
        {
            return _MB_SetFileOpenPwd(pwd) == EMBRetVal.MEMBRAIN_OK;
        }

        /// Remove the password for opening files
        static protected void MB_RemoveFileOpenPwd()
        {
            _MB_RemoveFileOpenPwd();
        }

        // -------------------- Functions for editing neural nets down from here ------------------------
        /// Clear selection
        static protected void MB_ClearSelection()
        {
            _MB_ClearSelection();
        }

        /// Select neurons by name. Return number of found (and selected) neurons
        static protected int MB_SelectNeuronsByName(string neuronName, bool addToSelection, bool findMultiple)
        {
            return _MB_SelectNeuronsByName(neuronName, addToSelection, findMultiple);
        }

        /// Select an input neuron
        static protected bool MB_SelectInput(int inNeuronIdx, bool addToSelection)
        {
            return (_MB_SelectInput(inNeuronIdx, addToSelection) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Select all input neurons
        static protected void MB_SelectAllInputs(bool addToSelection)
        {
            _MB_SelectAllInputs(addToSelection);
        }

        /// Select an output neuron
        static protected bool MB_SelectOutput(int outNeuronIdx, bool addToSelection)
        {
            return (_MB_SelectOutput(outNeuronIdx, addToSelection) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Select all output neurons
        static protected void MB_SelectAllOutputs(bool addToSelection)
        {
            _MB_SelectAllOutputs(addToSelection);
        }

        /// Get the number of hidden layers in the net
        static protected int MB_GetHiddenLayerCount()
        {
            return _MB_GetHiddenLayerCount();
        }

        /// Get the number of neurons in a given hidden layer
        static protected int MB_GetHiddenCount(int hidLayerIdx)
        {
            return _MB_GetHiddenCount(hidLayerIdx);
        }

        /// Get the number of neurons in all hidden layers
        static protected int MB_GetHiddenCountAll()
        {
            return _MB_GetHiddenCountAll();
        }

        /// Get the number of neurons in the context layer
        static protected int MB_GetContextCount()
        {
            return _MB_GetContextCount();
        }

        /// Get the number of neurons in the unresolved layer
        static protected int MB_GetUnresolvedCount()
        {
            return _MB_GetUnresolvedCount();
        }

        /// Select a hidden neuron
        static protected bool MB_SelectHidden(int hidLayerIdx, int hidNeuronIdx, bool addToSelection)
        {
            return (_MB_SelectHidden(hidLayerIdx, hidNeuronIdx, addToSelection) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Select all neurons in a hidden layer
        static protected bool MB_SelectHiddenLayer(int hidLayerIdx, bool addToSelection)
        {
            return (_MB_SelectHiddenLayer(hidLayerIdx, addToSelection) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Select all hidden neurons
        static protected void MB_SelectAllHidden(bool addToSelection)
        {
            _MB_SelectAllHidden(addToSelection);
        }

        /// Select a context neuron
        static protected bool MB_SelectContext(int neuronIdx, bool addToSelection)
        {
            return (_MB_SelectContext(neuronIdx, addToSelection) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Select all context neurons
        static protected void MB_SelectAllContexts(bool addToSelection)
        {
            _MB_SelectAllContexts(addToSelection);
        }

        /// Select an unresolved neuron
        static protected bool MB_SelectUnresolved(int neuronIdx, bool addToSelection)
        {
            return (_MB_SelectUnresolved(neuronIdx, addToSelection) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Select all unresolved neurons
        static protected void MB_SelectAllUnresolved(bool addToSelection)
        {
            _MB_SelectAllUnresolved(addToSelection);
        }

        /// Clear the Extra Selection
        static protected void MB_ClearExtraSelection()
        {
            _MB_ClearExtraSelection();
        }

        /// Apply Extra Selection to the current selection
        static protected void MB_ExtraSelection()
        {
            _MB_ExtraSelection();
        }

        /// Connect FROM Extra Selection
        static protected void MB_ConnectFromExtra()
        {
            _MB_ConnectFromExtra();
        }

        /// Connect TO Extra Selection
        static protected void MB_ConnectToExtra()
        {
            _MB_ConnectToExtra();
        }

        /// Add an input neuron to the net
        static protected void MB_AddInput(int posX, int posY, string name)
        {
            _MB_AddInput(posX, posY, name);
        }

        /// Add an output neuron to the net
        static protected void MB_AddOutput(int posX, int posY, string name)
        {
            _MB_AddOutput(posX, posY, name);
        }

        /// Add an unresolved hidden neuron to the net
        static protected void MB_AddHidden(int posX, int posY, string name)
        {
            _MB_AddHidden(posX, posY, name);
        }


        /// Get the properties of the currently selected neuron
        static protected bool MB_GetSelectedNeuronProp(out NeuralNet.SMBNeuronProp prop)
        {
            SMBNeuronPropNative native = new SMBNeuronPropNative();
            bool ok = (_MB_GetSelectedNeuronProp(out native) == EMBRetVal.MEMBRAIN_OK);

            string s = Marshal.SizeOf(typeof(SMBNeuronPropNative)).ToString();

            prop = new NeuralNet.SMBNeuronProp();

            if (ok)
            {
                prop.act = native.act;
                prop.inputFunc = (NeuralNet.EMBInputFunc)native.inputFunc;
                prop.actFunc = (NeuralNet.EMBActFunc)native.actFunc;
                prop.actIgnoreVal = native.actIgnoreVal;
                prop.actSustain = native.actSustain;
                prop.actThres = native.actThres;
                prop.allowTeacherOutputConnect = (EMBBoolean)native.allowTeacherOutputConnect == EMBBoolean.MEMBRAIN_TRUE;
                prop.displayAct = (EMBBoolean)native.displayAct == EMBBoolean.MEMBRAIN_TRUE;
                prop.displayName = (EMBBoolean)native.displayName == EMBBoolean.MEMBRAIN_TRUE;
                prop.expLogistic = native.expLogistic;
                prop.fireThresHi = native.fireThresHi;
                prop.fireThresLow = native.fireThresLow;
                prop.isPixel = (EMBBoolean)native.isPixel == EMBBoolean.MEMBRAIN_TRUE;
                prop.lockActThres = (EMBBoolean)native.lockActThres == EMBBoolean.MEMBRAIN_TRUE;
                prop.normRangeHigh = native.normRangeHigh;
                prop.normRangeLow = native.normRangeLow;
                prop.outputFireLevel = (NeuralNet.EMBOutputFireLevel)native.outputFireLevel;
                prop.outputRecovTime = native.outputRecovTime;
                prop.parmTanHyp = native.parmTanHyp;
                prop.leakage = native.leakage;
                prop.binDiffSlope = native.binDiffSlope;
                prop.useActIgnoreVal = (EMBBoolean)native.useActIgnoreVal == EMBBoolean.MEMBRAIN_TRUE;
                prop.useNormalization = (EMBBoolean)native.useNormalization == EMBBoolean.MEMBRAIN_TRUE;
                prop.width = native.width;
            }

            return ok;
        }

        /// Set the properties of the selected neurons
        static protected bool MB_SetSelectedNeuronProp(ref NeuralNet.SMBNeuronProp prop)
        {
            SMBNeuronPropNative native = new SMBNeuronPropNative();
            
            native.act = prop.act;
            native.inputFunc = (MB_INT)prop.inputFunc;
            native.actFunc = (MB_INT)prop.actFunc;
            native.actIgnoreVal = prop.actIgnoreVal;
            native.actSustain = prop.actSustain;
            native.actThres = prop.actThres;
            native.allowTeacherOutputConnect = prop.allowTeacherOutputConnect ? (int)EMBBoolean.MEMBRAIN_TRUE : (int)EMBBoolean.MEMBRAIN_FALSE;
            native.displayAct = prop.displayAct ? (int)EMBBoolean.MEMBRAIN_TRUE : (int)EMBBoolean.MEMBRAIN_FALSE;
            native.displayName = prop.displayName ? (int)EMBBoolean.MEMBRAIN_TRUE : (int)EMBBoolean.MEMBRAIN_FALSE;
            native.expLogistic = prop.expLogistic;
            native.fireThresHi = prop.fireThresHi;
            native.fireThresLow = prop.fireThresLow;
            native.isPixel = prop.isPixel ? (int)EMBBoolean.MEMBRAIN_TRUE : (int)EMBBoolean.MEMBRAIN_FALSE;
            native.lockActThres = prop.lockActThres ? (int)EMBBoolean.MEMBRAIN_TRUE : (int)EMBBoolean.MEMBRAIN_FALSE;
            native.normRangeHigh = prop.normRangeHigh;
            native.normRangeLow = prop.normRangeLow;
            native.outputFireLevel = (int)prop.outputFireLevel;
            native.outputRecovTime = prop.outputRecovTime;
            native.parmTanHyp = prop.parmTanHyp;
            native.leakage = prop.leakage;
            native.binDiffSlope = prop.binDiffSlope;
            native.useActIgnoreVal = prop.useActIgnoreVal ? (int)EMBBoolean.MEMBRAIN_TRUE : (int)EMBBoolean.MEMBRAIN_FALSE;
            native.useNormalization = prop.useNormalization ? (int)EMBBoolean.MEMBRAIN_TRUE : (int)EMBBoolean.MEMBRAIN_FALSE;
            native.width = prop.width;
                  
            bool ok = (_MB_SetSelectedNeuronProp(ref native) == EMBRetVal.MEMBRAIN_OK);

            return ok;
        }

        /// Delete the selected objects
        static protected void MB_DeleteSelectedObjects()
        {
            _MB_DeleteSelectedObjects();
        }

        /// Prepare for a new net
        static protected void MB_ClearNet()
        {
            _MB_ClearNet();
        }

        /// Set the name of the selected neurons    
        static protected void MB_SetSelectedNeuronName(string name)
        {
            _MB_SetSelectedNeuronName(name);
        }

        /// Move all selected neurons
        static protected void MB_MoveSelectedNeurons(int dX, int dY)
        {
            _MB_MoveSelectedNeurons(dX, dY);
        }

        /// Select all links from Extra Selection to Selection
        static protected void MB_SelectFromExtra()
        {
            _MB_SelectFromExtra();
        }

        /// Select all links from Selection to Extra Selection
        static protected void MB_SelectToExtra()
        {
            _MB_SelectToExtra();
        }


        /// Get the properties of the selected link
        static protected bool MB_GetSelectedLinkProp(out NeuralNet.SMBLinkProp prop)
        {
            SMBLinkPropNative native = new SMBLinkPropNative();
            bool ok = (_MB_GetSelectedLinkProp(out native) == EMBRetVal.MEMBRAIN_OK);

            prop = new NeuralNet.SMBLinkProp();

            if (ok)
            {
                prop.displayWeight = (EMBBoolean)native.displayWeight == EMBBoolean.MEMBRAIN_TRUE;
                prop.length = native.length;
                prop.lockWeight = (EMBBoolean)native.lockWeight == EMBBoolean.MEMBRAIN_TRUE;
                prop.weight = native.weight;
            }

            return ok;
        }

        /// Set the properties of the selected links
        static protected bool MB_SetSelectedLinkProp(ref NeuralNet.SMBLinkProp prop)
        {
            SMBLinkPropNative native = new SMBLinkPropNative();

            native.displayWeight = prop.displayWeight ? (int)EMBBoolean.MEMBRAIN_TRUE : (int)EMBBoolean.MEMBRAIN_FALSE;
            native.length = prop.length;
            native.lockWeight = prop.lockWeight ? (int)EMBBoolean.MEMBRAIN_TRUE : (int)EMBBoolean.MEMBRAIN_FALSE;
            native.weight = prop.weight;

            bool ok = (_MB_SetSelectedLinkProp(ref native) == EMBRetVal.MEMBRAIN_OK);

            return ok;
        }

        /// Get the position of the selected neuron
        static protected bool MB_GetSelectedNeuronPos(out int x, out int y)
        {
            return (_MB_GetSelectedNeuronPos(out x, out y) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Get the nearest grid point to a given point        
        static protected bool MB_GetGridPoint(int x, int y, out int gridX, out int gridY)
        {
            return (_MB_GetGridPoint(x, y, out gridX, out gridY) == EMBRetVal.MEMBRAIN_OK);
        }

        /// Get the adjusted grid width        
        static protected int MB_GetGridWidth()
        {
            return _MB_GetGridWidth();
        }

        /// Get a random value between 0 an 1       
        static protected double MB_Random()
        {
            return _MB_Random();
        } 



// =============================================================================================================================
    }

}
