﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;

namespace MemBrainLib
{
    public class Lesson : MB_DLL
    {
        /// <summary>
        ///  Available recording types
        /// </summary>
        public enum ERecordingType
        {
            REC_TYPE_ACT = 0,   /// Record activations
            REC_TYPE_OUT = 1    /// Record output values
        }
        
        /// <summary>
        /// Create a new lesson and add it to the lesson management
        /// </summary>
        public Lesson()
        {
            int idx = GetNextFreeDllIndex();
            int dllLessonCount = MB_GetLessonCount();

            // Set increased lesson count in dll if required
            if (idx >= dllLessonCount)
            {
                MB_SetLessonCount(idx + 1);
            }

            this.DllIndex = idx;

            if (idx >= Lessons.Count)
            {
                Lessons.Add(this);
            }
            else
            {
                Lessons[idx] = this;
            }

            this.DeleteAllPatterns();
        }

        /// <summary>
        /// Provate constructor used for special internal purposes where a new lesson has been created automatically
        /// via a dll call and the new lesson index is determined automatically and handed over to the Lesson constructor
        /// after this dll call
        /// </summary>
        /// <param name="dllIdx"></param>
        private Lesson(int dllIdx)
        {
            int dllLessonCount = MB_GetLessonCount();

            this.DllIndex = dllIdx;

            if (dllIdx >= Lessons.Count)
            {
                Lessons.Add(this);
            }
            else
            {
                Lessons[dllIdx] = this;
            }
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Lesson()
        {
            // A lesson with intermediate index may not be deleted from the dll. This means we 
            // may only set a reduced dll lesson count in case all lessons with higher dll index than
            // the deleted lesson are not in use anymore.
            // If we are not allowed to delete the lesson we want at least delete all patterns
            // in the lesson.
            // Also, we are not allowed to set the dll lesson count to 0.
            int instanceCount = Lessons.Count;

            // We may only reduce the lesson count if the index is > 0 (The first lesson, i.e. with index 0, 
            // can not be deleted in the dll).
            bool reduceLessonCount = (DllIndex > 0);

            if (reduceLessonCount)
            {
                // If there are used lessons existing with a higher Dll index we may not reduce the 
                // dll lesson count
                for (int i = instanceCount - 1; i > DllIndex; i--)
                {
                    if (Lessons[i] != null)
                    {
                        reduceLessonCount = false;
                        break;
                    }
                }
            }

            if (reduceLessonCount)
            {
                // New lesson count id exactly the Dll index of the deleted lesson
                MB_SetLessonCount(DllIndex);
                Lessons.RemoveRange(DllIndex, instanceCount - DllIndex);
            }
            else
            {
                // We are not allowed to reduce the lesson count in the dll. Thus, we just mark
                // this lesson to be invalid.
                Lessons[DllIndex] = null;
            }
        }

        /// <summary>
        /// Get function for property 'DllIdx'
        /// </summary>
        public int DllIdx
        {
            get { return DllIndex; }
        }

        /// <summary>
        /// Previously active lesson index in the dll
        /// </summary>
        static private int sPrevDllIndex = -1;

        /// <summary>
        /// Array for the lesson instances
        /// </summary>
        static private ArrayList Lessons = new ArrayList();
        
        
        /// <summary>
        /// Get the next free (unused) lesson dll index
        /// </summary>
        static private int GetNextFreeDllIndex()
        {
            int idx = 0;

            // Find the next lesson that is not in use or end up with new index
            for (idx = 0; idx < Lessons.Count; idx++)
            {
                if (Lessons[idx] == null)
                {
                    break;
                }
            }

            return idx;
        }

        /// <summary>
        /// Prepare the dll for an operation on this specific lesson instance
        /// </summary>
        private void PrepareDllForThis()
        {
            // Backup currently active lesson first to be able to restore it afterwards
            sPrevDllIndex = MB_GetSelectedLesson();

            bool ok = MB_SelectLesson(DllIndex);

            Debug.Assert(ok, "Unable to prepare dll for lesson idx " + DllIndex.ToString());
        }

        /// <summary>
        /// Restore the active lesson in the dll
        /// </summary>
        private void RestoreActiveLessonInDll()
        {
            bool ok = MB_SelectLesson(sPrevDllIndex);

            Debug.Assert(ok, "Unable to restore previous lesson idx " + sPrevDllIndex.ToString());
        }


        /// <summary>
        /// The lesson's index within the Dll
        /// </summary>
        private int DllIndex;





//####################### Methods for using the created lessons ####################################

        /// <summary>
        /// Load lesson from file
        /// </summary>
        /// <param name="filePath"></param>
        /// <returns></returns>
        public bool Load(string filePath)
        {
            PrepareDllForThis();
            
            bool success = MB_LoadLesson(filePath);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Import lesson from Sectioned CSV file
        /// </summary>
        /// <param name="filePath"></param>
        /// <returns></returns>
        public bool Import(string filePath)
        {
            PrepareDllForThis();

            bool success = MB_ImportLesson(filePath);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Import lesson from Raw CSV file
        /// </summary>
        /// <param name="filePath"></param>
        /// <returns></returns>
        public bool ImportRaw(string filePath)
        {
            PrepareDllForThis();

            bool success = MB_ImportLessonRaw(filePath);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Import lesson inputs only from Raw CSV file
        /// </summary>
        /// <param name="filePath"></param>
        /// <returns></returns>
        public bool ImportInputsRaw(string filePath)
        {
            PrepareDllForThis();

            bool success = MB_ImportLessonInputsRaw(filePath);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Import lesson outputs only from Raw CSV file
        /// </summary>
        /// <param name="filePath"></param>
        /// <returns></returns>
        public bool ImportOutputsRaw(string filePath)
        {
            PrepareDllForThis();

            bool success = MB_ImportLessonOutputsRaw(filePath);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Save lesson to file (overwrite existing file if exists)
        /// </summary>
        /// <returns></returns>
        public bool Save()
        {
            PrepareDllForThis();

            bool success = MB_SaveLesson();

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Save lesson to file using the specified file name
        /// </summary>
        /// <param name="filePath"></param>
        /// <returns></returns>
        public bool SaveAs(string filePath)
        {
            PrepareDllForThis();

            bool success = MB_SaveLessonAs(filePath);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Export lesson to Sectioned CSV file using the specified file name
        /// </summary>
        /// <param name="filePath"></param>
        /// <param name="maxCols"></param>
        /// <returns></returns>
        public bool Export(string filePath, int maxCols)
        {
            PrepareDllForThis();

            bool success = MB_ExportLesson(filePath, maxCols);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Export lesson to Raw CSV file using the specified file name
        /// </summary>
        /// <param name="filePath"></param>
        /// <param name="maxCols"></param>
        /// <returns></returns>
        public bool ExportRaw(string filePath, int maxCols)
        {
            PrepareDllForThis();

            bool success = MB_ExportLessonRaw(filePath, maxCols);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Export lesson inputs only to Raw CSV file using the specified file name
        /// </summary>
        /// <param name="filePath"></param>
        /// <param name="maxCols"></param>
        /// <returns></returns>
        public bool ExportInputsRaw(string filePath, int maxCols)
        {
            PrepareDllForThis();

            bool success = MB_ExportLessonInputsRaw(filePath, maxCols);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Export lesson outputs only to Raw CSV file using the specified file name
        /// </summary>
        /// <param name="filePath"></param>
        /// <param name="maxCols"></param>
        /// <returns></returns>
        public bool ExportOutputsRaw(string filePath, int maxCols)
        {
            PrepareDllForThis();

            bool success = MB_ExportLessonOutputsRaw(filePath, maxCols);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Set the number of inputs for the lesson
        /// </summary>
        /// <param name="count"></param>
        /// <returns></returns>
        public bool SetInputCount(int count)
        {
            PrepareDllForThis();

            bool success = MB_SetLessonInputCount(count);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Get the number of inputs for the lesson
        /// </summary>
        /// <param name="count"></param>
        /// <returns></returns>
        public int GetInputCount()
        {
            PrepareDllForThis();

            int count = MB_GetLessonInputCount();

            RestoreActiveLessonInDll();

            return count;
        }

        /// <summary>
        /// Set the number of outputs for the lesson
        /// </summary>
        /// <param name="count"></param>
        /// <returns></returns>
        public bool SetOutputCount(int count)
        {
            PrepareDllForThis();

            bool success = MB_SetLessonOutputCount(count);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Get the number of outputs for the lesson
        /// </summary>
        /// <param name="count"></param>
        /// <returns></returns>
        public int GetOutputCount()
        {
            PrepareDllForThis();

            int count = MB_GetLessonOutputCount();

            RestoreActiveLessonInDll();

            return count;
        }

        /// <summary>
        /// Set the name of the lesson input at index 'idx'
        /// </summary>
        /// <param name="idx"></param>
        /// <param name="name"></param>
        /// <returns></returns>
        public bool SetInputName(int idx, string name)
        {
            PrepareDllForThis();

            bool success = MB_SetLessonInputName(idx, name);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Get the name of the lesson input at index 'idx'
        /// </summary>
        /// <param name="idx"></param>
        /// <returns></returns>
        public string GetInputName(int idx)
        {
            PrepareDllForThis();

            string name;
            bool success = MB_GetLessonInputName(idx, out name);

            RestoreActiveLessonInDll();

            if (success)
            {
                return name.ToString();
            }
            else
            {
                return "";
            }
        }

        /// <summary>
        /// Set the name of the lesson output at index 'idx'
        /// </summary>
        /// <param name="idx"></param>
        /// <param name="name"></param>
        /// <returns></returns>
        public bool SetOutputName(int idx, string name)
        {
            PrepareDllForThis();

            bool success = MB_SetLessonOutputName(idx, name);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Get the name of the lesson output at index 'idx'
        /// </summary>
        /// <param name="idx"></param>
        /// <returns></returns>
        public string GetOutputName(int idx)
        {
            PrepareDllForThis();

            string name;
            bool success = MB_GetLessonOutputName(idx, out name);

            RestoreActiveLessonInDll();

            if (success)
            {
                return name.ToString();
            }
            else
            {
                return "";
            }
        }

        /// <summary>
        /// Set the value of a specific input of the currently active pattern of the lesson
        /// </summary>
        /// <param name="idx"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public bool SetPatternInput(int idx, double value)
        {
            PrepareDllForThis();

            bool success = MB_SetPatternInput(idx, value);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Get the value of a specific input of the currently active pattern of the lesson
        /// </summary>
        /// <param name="idx"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public bool GetPatternInput(int idx, out double value)
        {
            PrepareDllForThis();

            bool success = MB_GetPatternInput(idx, out value);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Set the value for a specific output of the currently active pattern of the lesson
        /// </summary>
        /// <param name="idx"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public bool SetPatternOutput(int idx, double value)
        {
            PrepareDllForThis();

            bool success = MB_SetPatternOutput(idx, value);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Get the value of a specific output of the currently active pattern of the lesson
        /// </summary>
        /// <param name="idx"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public bool GetPatternOutput(int idx, out double value)
        {
            PrepareDllForThis();

            bool success = MB_GetPatternOutput(idx, out value);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Select the given pattern to be the active one for this lesson
        /// </summary>
        /// <param name="idx"></param>
        /// <returns></returns>
        public bool SelectPattern(int idx)
        {
            PrepareDllForThis();

            bool success = MB_SelectPattern(idx);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Get the index of the active pattern of this lesson
        /// </summary>
        /// <param name="idx"></param>
        /// <returns></returns>
        public int GetSelectedPattern()
        {
            PrepareDllForThis();

            int idx = MB_GetSelectedPattern();

            RestoreActiveLessonInDll();

            return idx;
        }

        /// <summary>
        /// Delete the currently active pattern of this lesson
        /// </summary>
        /// <returns>true on success</returns>
        public bool DeletePattern()
        {
            PrepareDllForThis();

            bool success = MB_DeletePattern();

            RestoreActiveLessonInDll();

            return success;
        }


        /// <summary>
        /// Delete all patterns of this lesson
        /// </summary>
        /// <param name="idx"></param>
        /// <returns>true on success</returns>
        public bool DeleteAllPatterns()
        {
            PrepareDllForThis();

            int size = MB_GetLessonSize();
            bool success = true;

            for (int i = 0; i < size; i++)
            {
                success = success && MB_SelectPattern(0);
                success = success && MB_DeletePattern();
            }

            RestoreActiveLessonInDll();

            return success;
        }


        /// <summary>
        /// Add a pattern to the end of this lesson
        /// </summary>
        /// <param name="idx"></param>
        /// <returns></returns>
        public void AddPattern()
        {
            PrepareDllForThis();

            MB_AddPattern();

            RestoreActiveLessonInDll();
        }

        /// <summary>
        /// Get the size (number of patterns) of the lesson
        /// </summary>
        /// <returns></returns>
        public int GetSize()
        {
            PrepareDllForThis();

            int size = MB_GetLessonSize();

            RestoreActiveLessonInDll();

            return size;
        }

        /// <summary>
        /// Enable/Disable the output data section of the active lesson
        /// </summary>
        /// <param name="enable"></param>
        public void EnableLessonOutData(bool enable)
        {
            PrepareDllForThis();

            MB_EnableLessonOutData(enable ? 1 : 0);

            RestoreActiveLessonInDll();
        }

        /// <summary>
        /// Take over I/O count and names from the given neural net
        /// </summary>
        /// <param name="net"></param>
        /// <returns></returns>
        public bool NamesFromNet(NeuralNet net)
        {
            PrepareDllForThis();

            int prevNetIdx = MB_GetSelectedNet();
            bool success = MB_SelectNet(net.DllIdx);

            success = success && MB_NamesFromNet();

            // Restore the previously selected net
            success = success && MB_SelectNet(prevNetIdx);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Transfer the I/O count and names from this lesson to the given net neural net
        /// </summary>
        /// <param name="net"></param>
        /// <returns></returns>
        public bool NamesToNet(NeuralNet net)
        {
            PrepareDllForThis();

            int prevNetIdx = MB_GetSelectedNet();
            bool success = MB_SelectNet(net.DllIdx);

            success = success && MB_NamesToNet();

            // Restore the previously selected net
            success = success && MB_SelectNet(prevNetIdx);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Apply the currently active pattern of this lesson to the inputs of the net
        /// </summary>
        /// <param name="net"></param>
        /// <returns></returns>
        public bool ApplyPattern(NeuralNet net)
        {
            PrepareDllForThis();

            int prevNetIdx = MB_GetSelectedNet();
            bool success = MB_SelectNet(net.DllIdx);

            success = success && MB_ApplyPattern();

            // Restore the previously selected net
            success = success && MB_SelectNet(prevNetIdx);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Set the recording type. This function is static
        /// </summary>
        /// <param name="type"></param>
        public static void SetRecordingType(ERecordingType type)
        {
            MB_SetRecordingType((int)type);
        }

        /// <summary>
        /// Start recording to this lesson. Note that this will stop recording to other lessons.
        /// </summary>
        /// <param name="stepCount"></param>
        /// <returns></returns>
        public bool StartRecording(int stepCount)
        {
            return MB_StartRecording(DllIndex, stepCount);
        }

        /// <summary>
        /// Stop recording. This function is static
        /// </summary>
        public static void StopRecording()
        {
            MB_StopRecording();
        }

        /// <summary>
        /// Think on all patterns in this lesson using the given net
        /// </summary>
        /// <returns>true if successful</returns>
        public bool Think(NeuralNet net)
        {
            PrepareDllForThis();

            int prevNetIdx = MB_GetSelectedNet();
            bool success = MB_SelectNet(net.DllIdx);

            success = success && MB_ThinkLesson();

            // Restore the previously selected net
            success = success && MB_SelectNet(prevNetIdx);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Select this lesson as the active (and thus training) lesson
        /// </summary>
        public void SelectAsActive()
        {
            if (MB_GetSelectedLesson() != DllIdx)
            {
                MB_SelectLesson(DllIndex);
            }
        }

        /// <summary>
        /// Select this lesson as the net error lesson
        /// </summary>
        public void SelectAsNetError()
        {
            if (MB_GetSelectedNetErrLesson() != DllIdx)
            {
                MB_SelectNetErrLesson(DllIndex);
            }
        }

        /// <summary>
        /// Create an FFT lesson based on this lesson
        /// </summary>
        /// <param name="complex">Set to true for a complex valued lesson (real and imaginary parts). 
        /// Set to false for an absolute value FFT without phase information</param>
        /// <param name="inputsAreColumns">Set to true if the time series input data is represented in the columns of the lesson.
        /// Set to false if the time series are the rows of the lesson.</param>
        /// <param name="minFreqIdx">The lowest frequency index to be included in the new lesson. 
        /// Set to 0 for including all frequencies down to DC value.</param>
        /// <param name="maxFreqPoints">Maximum number of frequnecy points to include in the new FFT lesson</param>
        /// <returns>The newly created FFT lesson. null if function was unsuccessful</returns>
        public Lesson CreateFftLesson(bool complex, bool inputsAreColumns, int minFreqIdx, int maxFreqPoints)
        {
            Lesson ret = null;

            PrepareDllForThis();
            bool success = MB_CreateFftLesson(complex, inputsAreColumns, minFreqIdx, maxFreqPoints);
            if (success)
            { // A new lesson has been created in the dll automatically and added to the lesson editor.
              // Create a Lesson object for it now and return it.
                ret = new Lesson(MB_GetLessonCount() - 1);
            }

            RestoreActiveLessonInDll();

            return ret;
        }

        /// <summary>
        /// Get freqeuncy that corresponds to a frequency index in an FFT.
        /// </summary>
        /// <param name="freqIdx">The frequency index to request the frequency for</param>
        /// <param name="overallSampleTime">The overall sampling time of the time series that was basis for this FFT</param>
        /// <returns>The corresponding freqeuncy value</returns>
        public static double GetFftFrequency(int freqIdx, double overallSampleTime)
        {
            return MB_GetFftFrequency(freqIdx, overallSampleTime);
        }

        /// <summary>
        /// Create a new lesson with averaged inputs from the current lesson
        /// </summary>
        /// <param name="newInputDimension">The new input dimension for the resulting average lesson</param>
        /// <returns>The newly created averaged lesson. null if function was unsuccessful</returns>
        public Lesson CreateAverageLesson(int newInputDimension)
        {
            Lesson ret = null;

            PrepareDllForThis();
            bool success = MB_CreateAverageLesson(newInputDimension);
            if (success)
            { // A new lesson has been created in the dll automatically and added to the lesson editor.
                // Create a Lesson object for it now and return it.
                ret = new Lesson(MB_GetLessonCount() - 1);
            }

            RestoreActiveLessonInDll();

            return ret;
        }
		
		/// <summary>
        /// Set the password to be used when saving this lesson
        /// </summary>
        /// <param name="pwd">Password</param>
        /// <returns>true if successful</returns>
        public bool SetPassword(string pwd)
        {
            PrepareDllForThis();

            bool success = MB_SetLessonFilePwd(pwd);

            RestoreActiveLessonInDll();

            return success;
        }

        /// <summary>
        /// Remove the password for this lesson. Will take effect when the lesson is saved to file.
        /// </summary>
        public void RemovePassword()
        {
            PrepareDllForThis();

            MB_RemoveLessonFilePwd();

            RestoreActiveLessonInDll();
        }
    }
}
